package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ByteChannel;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Source;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter.PerCCDMetaDataProvider;
import org.lsst.ccs.daq.ims.channel.WritableIntChannel;
import org.lsst.ccs.daq.ims.channel.BadPixelDetector;
import org.lsst.ccs.utilities.ccd.CCD;

/**
 * A Callable which when called reads a single-source worth of data from the DAQ
 * and writes a set of FITS files.
 *
 * @author tonyj
 */
class SourceHandler implements Callable<FileList> {

    private static final Logger LOG = Logger.getLogger(SourceHandler.class.getName());

    private final Source source;
    private final static int BUFFER_SIZE = Integer.getInteger("org.lsst.ccs.subsystem.imagehandling.BufferSize", 1950720);
    private final ImageHandlingConfig config;
    private final RebNode rebNode;
    private final boolean isStreaming;
    private final CountDownLatch darkTime;
    private BadPixelDetector badPixelDetector;
    private final String imageName;

    /**
     * Create a new SourceHandler
     *
     * @param darkTime A latch to indicate dark time is available
     * @param source The source to read data from
     * @param config The configuration to use
     * @param rebNode The REB corresponding to the source.
     * @param isStreaming True if data is to be read using the DAQ streaming
     * interface.
     */
    SourceHandler(CountDownLatch darkTime, String imageName, Source source, ImageHandlingConfig config, RebNode rebNode, boolean isStreaming) {
        this.darkTime = darkTime;
        this.imageName = imageName;
        this.source = source;
        this.config = config;
        this.rebNode = rebNode;
        this.isStreaming = isStreaming;
    }

    @Override
    public FileList call() throws IOException {

        rebNode.incrementImageCount(imageName);
        Map<File, File> tmpMap = new HashMap<>();
        try {
            FileList result = new FileList();
            FitsIntWriter.FileNamer namer = (Map<String, Object> props) -> {
                File file = config.getFitsFile(props);
                // Note, this is the name the file will have (after it is closed and renamed)
                // rather than the temporary naem it has initially.
                props.put("OriginalFileName", file.getName());
                if (config.isUseTempFile()) {
                    File tmpFile = config.getTempFitsFile(props);
                    tmpMap.put(tmpFile, file);
                    return tmpFile;
                } else {
                    return file;
                }
            };
            PerCCDMetaDataProvider metaDataProvider = (CCD ccd) -> Collections.singletonList(rebNode.getFitsService().getFitsHeaderMetadataProvider(ccd.getUniqueId()));
            // Set up channels to write per amplifier data
            // TODO: Better handling of multiple exceptions
            IOException io = null;
            ReadThread readThread = (ReadThread) Thread.currentThread();
            // Note, when streaming the source meta-data is NOT available until the first chuck of data arrives, so we need to delay creating the FITS file until then. 
            // TODO: This has become rather ugly, should be cleaned up
            FitsIntWriter decompress;
            if (config.isCheckForBadPixels()) {
                decompress = isStreaming ? new FitsIntWriter(source.getImage(), rebNode.getReb(), namer) {
                    @Override
                    protected WritableIntChannel createPixelFilter(WritableIntChannel channel) {
                        badPixelDetector = new BadPixelDetector(channel, config.getBadPixelLowThreshold(), config.getBadPixelHighThreshold());
                        return badPixelDetector;
                    }
                } : new FitsIntWriter(source, rebNode.getReb(), rebNode.getFitsService().getHeaderSpecificationMap(), namer, metaDataProvider) {
                    @Override
                    protected WritableIntChannel createPixelFilter(WritableIntChannel channel) {
                        badPixelDetector = new BadPixelDetector(channel, config.getBadPixelLowThreshold(), config.getBadPixelHighThreshold());
                        return badPixelDetector;
                    }
                };
            } else {
                decompress = isStreaming ? new FitsIntWriter(source.getImage(), rebNode.getReb(), namer) : new FitsIntWriter(source, rebNode.getReb(), rebNode.getFitsService().getHeaderSpecificationMap(), namer, metaDataProvider);
            }
            try (ByteChannel readChannel = source.openChannel(readThread.getStore(), isStreaming ? Source.ChannelMode.STREAM : Source.ChannelMode.READ)) {
                ByteBuffer bb = ByteBuffer.allocateDirect(BUFFER_SIZE);
                bb.order(ByteOrder.LITTLE_ENDIAN);
                Semaphore semaphore = readThread.getSemaphore();
                int totalBytes = 0;
                for (;;) {
                    bb.clear();
                    semaphore.acquire();
                    try {
                        int l = readChannel.read(bb);
                        if (l < 0) {
                            break;
                        }
                        totalBytes += l;
                    } finally {
                        semaphore.release();
                    }
                    bb.flip();
                    if (!decompress.isInitialized()) {
                        // We need to make sure the darkTime has arrived before proceeding to write the FITS headers
                        waitForDarkTime();
                        decompress.completeInitialization(source, rebNode.getFitsService().getHeaderSpecificationMap(), metaDataProvider);
                    }
                    decompress.write(bb.asIntBuffer());
                }
                if (badPixelDetector != null) {
                    // Note, we do not raise the alert here, we leave that for focal-plane when it receives the filelist
                    final int badPixels = badPixelDetector.getBadPixels();
                    if (badPixels > config.getBadPixelWarningLevel()) {
                        Level level = badPixels > config.getBadPixelAlarmLevel() ? Level.SEVERE : Level.WARNING;
                        result.addBadPixels(badPixels);
                        LOG.log(level, () -> String.format("Image %s source %s has %d bad pixels", source.getImage().getMetaData().getName(), source.getLocation(), badPixels));
                    }
                }
                int finalTotalBytes = totalBytes * 16 / 9;
                if (finalTotalBytes != decompress.getExpectedDataLength()) {
                    //TODO: This should result in an exception
                    LOG.log(Level.WARNING, () -> String.format("Data length did not match expected (%,d != %,d)", finalTotalBytes, decompress.getExpectedDataLength()));
                }
            } catch (InterruptedException x) {
                io = new InterruptedIOException("Interrupt during IO");
                io.initCause(x);
                throw io;
            } catch (IOException x) {
                io = x;
                throw io;
            } finally {
                List<File> filesToBeWritten = decompress.getFiles();
                IOException closeException = null;
                try {
                    decompress.close();
                } catch (IOException x) {
                    closeException = x;
                }

                if (filesToBeWritten != null) {
                    if (closeException != null || io != null) {
                        for (File file : filesToBeWritten) {
                            // Note: This does not throw an exception on failure
                            file.delete();
                        }
                    } else if (config.isUseTempFile()) {
                        for (File tempFile : filesToBeWritten) {

                            File finalName = tmpMap.get(tempFile);
                            boolean success = tempFile.renameTo(finalName);
                            if (!success) {
                                /// This should be exceedingly rare, so we will not worry about cleanup in this case
                                throw new IOException("Unable to rename " + tempFile + " to " + finalName);
                            }
                            result.add(finalName);
                        }
                        // TODO: Clean up temp file directory if it is empty?
                    } else {
                        result.addAll(filesToBeWritten);
                    }
                }
                if (io == null && closeException != null) {
                    throw closeException;
                }
            }
            return result;
        } catch (IOException | DAQException ioe) {
            throw new IOException("Exception for " + rebNode.getReb().getFullName(), ioe);
        } finally {
            rebNode.decrementImageCount(imageName);
            rebNode.getFitsService().clearNonStickyHeaderKeywordValues();
        }
    }

    private void waitForDarkTime() throws InterruptedException, IOException {
        if (darkTime.getCount() != 0) {
            long start = System.nanoTime();
            boolean ok = darkTime.await(10, TimeUnit.SECONDS);
            if (!ok) {
                throw new IOException("Timeout waiting for darkTime");
            }
            LOG.log(Level.FINE, () -> String.format("Waited %,dns for darkTime", System.nanoTime() - start));
        }
    }
}
