package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import java.util.Map;
import java.util.stream.Collectors;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.pattern.FileNamePatternUtils;

/**
 *
 * @author tonyj
 */
@SuppressWarnings("FieldMayBeFinal")
public class ImageHandlingConfig implements ConfigurationBulkChangeHandler {

    private static final String TMP_POSTFIX = ".$tmp$";

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile String FITSRootDirectory = System.getProperty("java.io.tmpdir");

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile String FITSDirectoryPattern = "${ImageDate}/${ImageName}";

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile String FITSFilePattern = "${ImageName}_${RaftBay}_${CCDSlot}.fits";

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile boolean FITSAutoSave = true;

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile boolean useTempFile = true;

    @ConfigurationParameter(category = "FitsHandling", units = "unitless")
    private volatile String tempFileRelativeLocation = ".";

    @ConfigurationParameter(category = "FitsHandling", units = "unitless", maxLength = 10)
    @SuppressWarnings("VolatileArrayField")
    private volatile String[] subsystemsToClearOnNewRun = { "bot-bench", "ccob", "bot-motorpltform" };

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile String daqPartition = "dev";

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile String daqFolder = "raw";

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile String locations = "R22/Reb0,R22/Reb1,R22/Reb2";

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile int daqThreads = 4;

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile boolean useStreaming = false;
    
    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile boolean checkForBadPixels = false;

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile int badPixelLowThreshold = 10000;
    
    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile int badPixelHighThreshold = 240000;

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile int badPixelWarningLevel = 0;

    @ConfigurationParameter(category = "DAQ", units = "unitless")
    private volatile int badPixelAlarmLevel = 20000;
    
    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        String dir = parametersView.get("FITSRootDirectory").toString();
        File dirFile = new File(dir);
        dirFile.mkdirs();
        if (!dirFile.isDirectory()) {
            throw new IllegalArgumentException("FITSRootDirectory: Not a directory");
        }
        if (!dirFile.canWrite()) {
            throw new IllegalArgumentException("FITSRootDirectory: Not writable");
        }
        // Validate that the locations syntax is valid.
        LocationSet.of(parametersView.get("locations").toString());
        // TODO: Also verify that the locations are a subset of the locations 
        // specified in the geometry/fits handlers.
    }

    public String getFITSRootDirectory() {
        return FITSRootDirectory;
    }

    public String getFITSDirectoryPattern() {
        return FITSDirectoryPattern;
    }

    public String getFITSFilePattern() {
        return FITSFilePattern;
    }

    public boolean isFITSAutoSave() {
        return FITSAutoSave;
    }

    public String getDaqPartition() {
        return daqPartition;
    }

    LocationSet getLocationsToProcess() {
        return LocationSet.of(locations);
    }

    public String getDaqFolder() {
        return daqFolder;
    }

    public int getDaqThreads() {
        return daqThreads;
    }

    public boolean isUseTempFile() {
        return useTempFile;
    }

    public boolean isUseStreaming() {
        return useStreaming;
    }

    File getFitsFile(Map<String, Object> data) {
        // Convert input data to a map of String, String TODO: Move this to FileNamePatternUtils
        Map<String, String> props = data.entrySet().stream().collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue().toString()));
        File root = new File(FITSRootDirectory);
        String fileDir = FileNamePatternUtils.resolveFileName(FITSDirectoryPattern, props);
        File dir = new File(root, fileDir);
        String fileName = FileNamePatternUtils.resolveFileName(FITSFilePattern, props);
        File result = new File(dir, fileName);
        result.getParentFile().mkdirs();
        return result;
    }

    File getTempFitsFile(Map<String, Object> data) {
        // Convert input data to a map of String, String TODO: Move this to FileNamePatternUtils
        Map<String, String> props = data.entrySet().stream().collect(Collectors.toMap(e -> e.getKey(), e -> e.getValue().toString()));
        File root = new File(FITSRootDirectory);
        String fileDir = FileNamePatternUtils.resolveFileName(FITSDirectoryPattern, props);
        File dir = new File(root, fileDir);
        dir = new File(dir, tempFileRelativeLocation);
        String fileName = FileNamePatternUtils.resolveFileName(FITSFilePattern, props);
        fileName += TMP_POSTFIX;
        File result = new File(dir, fileName);
        result.getParentFile().mkdirs();
        return result;
    }

    String[] getSubsystemsToClear() {
        return subsystemsToClearOnNewRun;
    }

    public int getBadPixelLowThreshold() {
        return badPixelLowThreshold;
    }

    public int getBadPixelHighThreshold() {
        return badPixelHighThreshold;
    }

    public int getBadPixelWarningLevel() {
        return badPixelWarningLevel;
    }

    public int getBadPixelAlarmLevel() {
        return badPixelAlarmLevel;
    }

    public boolean isCheckForBadPixels() {
        return checkForBadPixels;
    }
}
