package org.lsst.ccs.subsystem.imagehandling;

import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.Semaphore;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;
import org.lsst.ccs.utilities.location.LocationSet;

public class ImageHandlingClient implements HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageHandlingClient.class.getName());

    private ExecutorService executor;

    @LookupPath
    private String path;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageHandlingConfig imageHandlingConfig;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final List<RebNode> rebs = new ArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    @Override
    public void build() {
    }

    @Override
    public void start() {
        LOG.log(Level.FINE, "Starting ImageHandling client {0})", path);
    }

    @Override
    public void postStart() {
        try {
            Semaphore semaphore = new Semaphore(imageHandlingConfig.getDaqThreads());
            int nLocations = imageHandlingConfig.getLocationsToProcess().size();
            // pre-create all the stores we will need
            ConcurrentLinkedDeque<Store> stores = new ConcurrentLinkedDeque<>();
            for (int i=0; i<nLocations+2; i++) {
                stores.add(new Store(imageHandlingConfig.getDaqPartition()));
            }
            ThreadFactory readThreadFactory = (Runnable r) -> new ReadThread(r, stores, semaphore);

            executor = new ThreadPoolExecutor(nLocations+2, nLocations+2, 60L,
                    TimeUnit.SECONDS, new SynchronousQueue<>(), readThreadFactory);

            Store store = new Store(imageHandlingConfig.getDaqPartition());
            store.addImageListener(new ImageListener() {
                @Override
                public void imageCreated(Image image) {
                    if (imageHandlingConfig.isUseStreaming()) {
                        if (!checkFolder(image)) return;
                        handleImage(image, true);
                    }
                }

                @Override
                public void imageComplete(Image image) {
                    KeyValueData kvd = new KeyValueData(ImageReceivedEvent.EVENT_KEY, new ImageReceivedEvent(image.getMetaData()));
                    agent.publishSubsystemDataOnStatusBus(kvd);
                    if (!imageHandlingConfig.isUseStreaming()) {
                        if (!checkFolder(image)) return;
                        handleImage(image, false);
                    }
                }

                private void handleImage(Image image, boolean isStreaming) {
                    LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
                    locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
                    Future<FileList> future = executor.submit(new ImageHandler(image, executor, imageHandlingConfig, rebs, isStreaming));
                    executor.submit(new ImageEventSender(future, agent, locationsWritten, image.getMetaData().getName()));
                }

                private boolean checkFolder(Image image) {
                    // Check if image is in expected folder
                    final String daqFolder = imageHandlingConfig.getDaqFolder();
                    if (daqFolder != null && !"".equals(daqFolder)) {
                        if (!imageHandlingConfig.getDaqFolder().equals(image.getMetaData().getCreationFolderName())) {
                            return false;
                        }
                    }
                    return true;
                }
            });
        } catch (DAQException x) {
            // TODO: Is this the right thing to do? unfortunately HasLifecycle 
            // javadoc does not say what happens if an exception is thrown.
            throw new RuntimeException("Failed to connect to DAQ", x);
        }
    }

    ExecutorService getExecutorService() {
        return executor;
    }

    FileList fetchImage(String imageName) throws DAQException, InterruptedException, ExecutionException {
        Store store = new Store(imageHandlingConfig.getDaqPartition());
        final String daqFolder = imageHandlingConfig.getDaqFolder();
        Folder imageFolder = store.getCatalog().find(daqFolder);
        if (imageFolder == null) {
            throw new RuntimeException("Folder " + daqFolder + " not found");
        }
        Image image = imageFolder.find(imageName);
        if (image == null) {
            throw new RuntimeException("Image " + imageName + " not found");
        }
        LocationSet locationsWritten = new LocationSet(image.getMetaData().getLocations());
        locationsWritten.retainAll(imageHandlingConfig.getLocationsToProcess());
        Future<FileList> future = executor.submit(new ImageHandler(image, executor, imageHandlingConfig, rebs, false));
        return future.get();
    }
}
