package org.lsst.ccs.subsystem.imagehandling;

import java.io.File;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ByteChannel;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import nom.tam.fits.FitsException;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Source;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter;
import org.lsst.ccs.daq.ims.channel.FitsIntWriter.PerCCDMetaDataProvider;
import org.lsst.ccs.subsystem.imagehandling.states.ImageHandlingState;
import org.lsst.ccs.utilities.ccd.CCD;

/**
 *
 * @author tonyj
 */
public class SourceHandler implements Callable<FileList> {

    private final Source source;
    private final static int BUFFER_SIZE = 1024 * 1024;
    private final ImageHandlingConfig config;
    private final RebNode rebNode;

    SourceHandler(Source source, ImageHandlingConfig config, RebNode rebNode) {
        this.source = source;
        this.config = config;
        this.rebNode = rebNode;
    }

    @Override
    public FileList call() throws IOException, DAQException, FitsException {
        
        rebNode.updateState(ImageHandlingState.READING);
        try {
            FileList result = new FileList();
            FitsIntWriter.FileNamer namer = (Map<String, Object> props) -> {
                File file = config.getFitsFile(props);
                result.add(file);
                return file;
            };
            PerCCDMetaDataProvider metaDataProvider = (CCD ccd) -> Collections.singletonList(rebNode.getFitsService().getFitsHeaderMetadataProvider(ccd.getUniqueId()));
            // Set up channels to write per amplifier data
            try (FitsIntWriter decompress = new FitsIntWriter(source, rebNode.getReb(), rebNode.getFitsService().getHeaderSpecificationMap(), namer, metaDataProvider);
                    ByteChannel readChannel = source.openChannel(Source.ChannelMode.READ)) {
                ByteBuffer bb = ByteBuffer.allocateDirect(BUFFER_SIZE);
                bb.order(ByteOrder.LITTLE_ENDIAN);
                for (;;) {
                    int l = readChannel.read(bb);
                    if (l < 0) {
                        break;
                    }
                    bb.flip();
                    decompress.write(bb.asIntBuffer());
                    bb.clear();
                }
            }
            return result;
        } finally {
            rebNode.updateState(ImageHandlingState.IDLE);
        }
    }
}
