package org.lsst.ccs.subsystem.imagehandling.data;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.zip.CheckedOutputStream;
import java.util.zip.Checksum;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.imagenaming.ImageName;

/**
 * An object representing an additional data file to be written into the image
 * output directory by the "primary" image handler, and sent to USDF.
 *
 * @author tonyj
 */
@SkipEncoding
public abstract class AdditionalFile implements Serializable {

    private final String fileName;
    private final String fileType;
    private final ImageName obsId;
    // I have that this is a float, but this is what the LFA wants
    private float version;
    private final String mimeType;
    private static final long serialVersionUID = 1;
    public static final String EVENT_KEY = "additionalFile";
    static final float DEFAULT_VERSION = 1.0f;

    public AdditionalFile(String fileName, String fileType, ImageName obsId, float version, String mimeType) {
        this.fileName = fileName;
        this.obsId = obsId;
        this.fileType = fileType;
        this.version = version;
        this.mimeType = mimeType;
    }

    public AdditionalFile(String fileName, String fileType, ImageName obsId, float version) {
        this(fileName, fileType, obsId, version, null);
    }

    public AdditionalFile(String fileName, String fileType, ImageName obsId) {
        this(fileName, fileType, obsId, DEFAULT_VERSION, null);
    }

    public String getFileName() {
        return fileName;
    }

    public ImageName getObsId() {
        return obsId;
    }

    public String getFileType() {
        return fileType;
    }

    public float getVersion() {
        return version == 0 ? DEFAULT_VERSION : version;
    }

    public void setVersion(float version) {
        this.version = version;
    }

    public String getMimeType() {
        if (mimeType == null) {
            String guessedMimeType = TextFile.MIME_TYPE;
            if (fileName.endsWith("json")) {
                guessedMimeType = JsonFile.MIME_TYPE;
            } else if (fileName.endsWith("csv")) {
                guessedMimeType = ECSVFile.MIME_TYPE;
            }
            return guessedMimeType;
        } else {
            return mimeType;
        }
    }
    /**
     * Compute the size and checksum of the file
     * @param cksum32 The Checksum to use (will be filled on return)
     * @return The file length in bytes
     * @throws IOException
     */
    public long computeFileSizeAndChecksum(Checksum cksum32) throws IOException {
        try (CountOutputStream cnos = new CountOutputStream(); CheckedOutputStream cos = new CheckedOutputStream(cnos, cksum32)) {
            writeFile(cos);
            cos.flush();
            return cnos.getByteCount();
        }
    }

    public abstract void writeFile(OutputStream out) throws IOException;

    @Override
    public String toString() {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            writeFile(baos);
            baos.close();
            return baos.toString();
        } catch (IOException x) {
            return x.toString();
        }
    }

    private class CountOutputStream extends OutputStream {

        long byteCount = 0;

        @Override
        public void write(int b) throws IOException {
            byteCount++;
        }

        long getByteCount() {
            return this.byteCount;
        }

    }

}
