package org.lsst.ccs.subsystem.imagehandling.data;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicInteger;
import org.lsst.ccs.bus.annotations.SkipEncoding;

/**
 * A simple extension of <code>List&lt;File&gt;</code> which adds an extra
 * utility method for getting the common root directory of all the files in the
 * list.
 */
@SkipEncoding
public class FileList extends ConcurrentSkipListSet<File> implements Serializable {

    private static final long serialVersionUID = 3543276216049016663L;
    private AtomicInteger badPixelCount = new AtomicInteger(0);

    // Handle old versions which did not have badPixelCount
    private void readObject(ObjectInputStream stream)
            throws IOException, ClassNotFoundException {  
        stream.defaultReadObject();
        if (badPixelCount == null) {
           badPixelCount = new AtomicInteger(0);
        }
    }  

    public File getCommonParentDirectory() {
        Iterator<File> iterator = this.iterator();
        if (!iterator.hasNext()) {
            return new File("/");
        }
        File result = iterator.next().getParentFile();
        while (iterator.hasNext()) {
            File file = iterator.next();
            while (!file.getAbsolutePath().startsWith(result.getAbsolutePath())) {
                result = result.getParentFile();
            }
        }
        return result;
    }

    @Override
    public String toString() {
        Path parent = getCommonParentDirectory().toPath();
        List<Path> relativePaths = new ArrayList();
        for (File file : this) {
            relativePaths.add(parent.relativize(file.toPath()));
        }
        String badPixels = badPixelCount.get() > 0 ? String.format(" (%d bad pixels)", badPixelCount.get()) : "";
        return String.format("%d files%s: %s %s", size(), badPixels, parent, relativePaths);
    }

    public void addBadPixels(int badPixels) {
        badPixelCount.addAndGet(badPixels);
    }

    public int getBadPixelCount() {
        return badPixelCount.get();
    }

    public boolean addAll(FileList fileList) {
        badPixelCount.addAndGet(fileList.getBadPixelCount());
        return super.addAll(fileList);
    }

}
