package org.lsst.ccs.subsystem.focalplane;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A DerivedChannel that keeps a running average of the provided list of
 * channels.
 * 
 * Configuration parameter temperatureNames contains the list of the temperature
 * channels to be used in evaluating the running average. The names are meant to be
 * those of SIBLING channels; this means that only the leaf name should be provided (Temp6, Temp10).
 * 
 * To start the evaluation of the running average invoke method startAveraging; to stop
 * the evaluation, invoke endAveraging.
 * 
 * When the evaluation is stopped the channel publishes Double.NaN.
 *
 * @author The LSST CCS Team
 */
public class TemperatureRunningAverageChannel extends DerivedChannel implements StateChangeListener {

    private static final Logger LOG = Logger.getLogger(TemperatureRunningAverageChannel.class.getName());

    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private Map<String, Channel> channelMap = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Monitor monitor;    

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;    

    @ConfigurationParameter(name = "temperatureNames", category = "General",
            description = "List of sibling temperature channels to average",
            maxLength = 5, units = "unitless")
    protected volatile List<String> temperatureNames = new ArrayList<>();
    
    private volatile double runningSum = 0;
    private volatile int numberOfSamples = 0;

    private volatile List<Channel> tempChannels = new ArrayList();

    private volatile boolean evaluateAverage = false;

    private final Object lock = new Object();
    
    private MonitorUpdateTask thisChannelTask;

    @Override
    public void init() {
        super.init(); 
        thisChannelTask = monitor.getMonitorUpdateTasksForChannels(this).get(0);
        agentStateService.addStateChangeListener(this, FocalPlaneState.class);
    }

    @Override
    public void validateBulkChange(Map<String, Object> params) {
        super.validateBulkChange(params);
        synchronized(lock) {
            if ( evaluateAverage ) {
                throw new RuntimeException("Cannot change the list of temperature while the average is being evaluated.");
            }
            List<String> channelList = (List<String>) params.get("temperatureNames");

            StringBuilder sb = new StringBuilder();
            sb.append("Updated list of temperatures to average for channel ").append(getPath()).append("\n");

            List tmpTempChannels = new ArrayList();

            for (String channelName : channelList) {
                Channel c = channelMap.get(channelName);
                if (c == null) {
                    throw new IllegalArgumentException("Channel name " + channelName + " is invalid. No Channel was found.");
                }
                tmpTempChannels.add(c);
                sb.append(c.getPath()).append(" ");
            }
            tempChannels = tmpTempChannels;
            if ( tempChannels.isEmpty() ) {
                sb.append("No temperature channels were selected.");
            }
            LOG.log(Level.FINE, sb.toString());
        }
    }

    public void stateChanged(CCSTimeStamp stateTransitionTimestamp, Object changedObj, Enum<?> newState, Enum<?> oldState) {
        if ( newState == FocalPlaneState.INTEGRATING ) {
            startAveraging();
        } else {
            endAveraging();
        }
        
    }
    
    
    
    private void resetAverage() {
        runningSum = 0;
        numberOfSamples = 0;
    }

    /**
     * Invoke this method to start evaluating a running average of the selected
     * temperature channels over the provided duration.
     * 
     */
    private void startAveraging() {
        synchronized (lock) {
            evaluateAverage = true;
        }
    }

    /**
     * Invoke this method stop evaluating the running average.
     * This will reset the average to zero and the channel will publish Double.NaN
     * 
     */
    private void endAveraging() {
        synchronized (lock) {
            evaluateAverage = false;
            resetAverage();
            thisChannelTask.resetForcedDataPublication();
        }
    }

    @Override
    public double evaluateDerivedValue() {
        synchronized (lock) {
            if (!evaluateAverage || tempChannels.isEmpty()) {
                return Double.NaN;
            }
            thisChannelTask.forceDataPublicationOnNextUpdates(2);
            for (Channel c : tempChannels) {
                double val = c.getValue();
                if ( Double.isFinite(val) ) {
                    runningSum += val;
                    numberOfSamples++;
                }
            }
            return runningSum / numberOfSamples;
        }
    }

}
