package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * A CCSImageList is a playlist backed by a simple ASCII text file.
 * Such playlists can be used both by CCSEmulatedPlayList or CCSSimulatedPlayList
 * @author tonyj
 */
public class CCSImageList {


    private final Path path;
    private final String dataFolder;
    private final String daqFolder;
    private final List<String> images;

    /**
     * Create a CCSImageList
     * @param path The path to the .plist file
     * @param dataFolder The data folder containing .raw and .meta files, or "DAQ". 
     * @param daqFolder The folder in the 2-day store containing the images
     * @param images The list of image names.
     */
    CCSImageList(Path path, String dataFolder, String daqFolder, List<String> images) {
        this.path = path;
        this.dataFolder = dataFolder;
        this.daqFolder = daqFolder;
        this.images = images;
    }
   
    void save() throws IOException {
        try (PrintWriter out = new PrintWriter(Files.newBufferedWriter(path, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING))) {
            out.println(dataFolder);
            out.println(daqFolder);
            for (String image : images) {
                out.println(image);
            }  
        } 
    }
    
    /**
     * Get the size of the image list
     * @return The size
     */
    int size() {
        return images.size();
    }
    
    /**
     * Add an image to the image list
     * @param image The image name. No check is performed that the image actually exists.
     */
    void add(String image) {
        images.add(image);
    }
    
    /**
     * Reads a playlist from a .plist file.
     * @param path The path to the .plist file
     * @return The created CCSImageList
     * @throws IOException If the file cannot be read successfully.
     */
    static CCSImageList open(Path path) throws IOException {
        List<String> lines = Files.readAllLines(path);
        // The first line of the file is DAQ for emulated playlists, or the folder containing
        // the .raw and .meta files for simulated playlists.
        String dataFolder = lines.get(0);
        // The DAQ 2-day store folder containing the raw data
        String daqFolder = lines.get(1);
        // The list of image names, one per line.
        List<String> imageList = lines.subList(2, lines.size());
        return new CCSImageList(path, dataFolder, daqFolder, imageList);
    }

    /**
     * Create a new (empty) image list
     * @param playListPath The path where the ,plist will be stored 
     * @param dataFolder The corresponding data folder (or "DAQ")
     * @param daqFolder The daq 2-day store folder.
     * @return new newly created empty image list
     */
    static CCSImageList create(Path playListPath, String dataFolder, String daqFolder) {
        List<String> imageList = new ArrayList<>();
        return new CCSImageList(playListPath, dataFolder, daqFolder, imageList);
    }    
    
    String getDAQFolder() {
        return daqFolder;
    }

    String getName() {
        return path.getFileName().toString();
    }

    String get(int index) {
        return images.get(index);
    }

    public String getDataFolder() {
        return dataFolder;
    }

    List<String> getImages() {
        return Collections.unmodifiableList(images);
    }
   
}
