package org.lsst.ccs.subsystem.focalplane;

import java.time.Duration;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.AutoCloseableReentrantLock;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.PeriodicTaskExceptionHandler;
import org.lsst.ccs.utilities.scheduler.Scheduler;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Collects (almost) all idle clear related functionality.
 * @author tonyj
 */
class IdleClear {

    private static final Logger LOG = Logger.getLogger(IdleClear.class.getName());
    
    private final SequencerConfig sequencerConfig;
    private final IdleClearSupport sequencers;
    private ScheduledFuture<Void> scheduledIdleClear;
    // If idle clear has been paused, this will be non-null, else null
    private volatile ScheduledFuture<?> scheduledIdleClearPause;
    //Periodic task responsible to run idleClear, null if not active
    private PeriodicTask idleClearPeriodicTask;
    // scheduler which starts idle clear after quiescent for a certain time.
    private final Scheduler idleClearScheduler;        
    private final AtomicBoolean idleClearLock = new AtomicBoolean(false);
    private final AgentStateService agentStateService;

    /**
     * Create IDLE_CLEAR
     * @param sequencerConfig The sequencer configuration
     * @param subsys The subsystem, may be <code>null</code> for testing
     * @param sequencers The real Sequencers, or a dummy implementation of IdleClearSupport for testing
     */
    public IdleClear(SequencerConfig sequencerConfig, FocalPlaneSubsystem subsys, IdleClearSupport sequencers) {
        this.sequencerConfig = sequencerConfig;
        this.sequencers = sequencers;
        idleClearScheduler = subsys == null ? new Scheduler("IdleClearScheduler", 1) : subsys.getScheduler();
        
        this.agentStateService = subsys != null ? subsys.getAgentService(AgentStateService.class) : null;
        
        if (agentStateService != null) {
            agentStateService.addStateChangeListener(new StateChangeListener() {
                @Override
                public void stateChanged(CCSTimeStamp transitionTime, Object changedObj, Enum<?> newState, Enum<?> oldState) {
                    if (newState == FocalPlaneState.QUIESCENT) {
                        focalPlaneEnteredQuiescent(agentStateService);
                    } else {
                        subsys.getSequencers().getIdleClear().focalPlaneLeftQuiescent();
                    }
                }
            }, FocalPlaneState.class);
        }
    }
    
    private void focalPlaneEnteredQuiescent(AgentStateService agentStateService) {
        long idleClearTimeout = sequencerConfig.getIdleClearTimeout();
        if (idleClearTimeout >= 0 && scheduledIdleClearPause == null) {

            scheduledIdleClear = idleClearScheduler.schedule(() -> {
                synchronized (agentStateService.getStateLock()) {
                    if (agentStateService.isInState(FocalPlaneState.QUIESCENT)) {
                        startIdleClear();
                    }
                }
                return null;
            }, idleClearTimeout, TimeUnit.MILLISECONDS);
        }
    }
    
    private void focalPlaneLeftQuiescent() {
        cancelScheduledIdleClear();
    }
   
    /**
     * Stop any current idle clear, and pause future idle clears for given time period
     * @param pauseDuration 
     */
    private synchronized void pause(Duration pauseDuration) throws REBException, DAQException, RaftException {
        // Schedule the pause
        scheduledIdleClearPause = idleClearScheduler.schedule(() -> restartIdleClearAfterPause(), pauseDuration.toMillis(), TimeUnit.MILLISECONDS);

        // Cancel any scheduledIdleClear, and don't start any more until pause is done
        cancelScheduledIdleClear();
        // Stop the idle clear if it is running
        stopIdleClearIfRunning();
    }

    private synchronized void cancelPauseAndRestart() throws REBException, RaftException, DAQException {
        if (scheduledIdleClearPause != null) {
            scheduledIdleClearPause.cancel(false);
            scheduledIdleClearPause = null;
            startIdleClear();
        }
    }
    
    private synchronized void restartIdleClearAfterPause() {
        if (scheduledIdleClearPause != null) {
            scheduledIdleClearPause = null;
            try {
                startIdleClear();
            } catch (RaftException | REBException | DAQException x) {
                LOG.log(Level.SEVERE, "Unable to restart idle clear after pause", x);
            }
        }
    }

    private synchronized void stopIdleClearIfRunning() throws REBException, DAQException, RaftException {
        synchronized (idleClearLock) {
            boolean wasSet = idleClearLock.getAndSet(false);
            if (wasSet) {
                cancelPeriodicTask();
                sequencers.terminateIdleClear(FocalPlaneState.QUIESCENT);
            }
        }
    }

    private void cancelPeriodicTask() {
        // Stop any more tasks from being scheduled
        if (idleClearPeriodicTask != null) {
            idleClearPeriodicTask.cancel(false);
            idleClearPeriodicTask = null;
        }
    }

    private synchronized void cancelScheduledIdleClear() {
        if (scheduledIdleClear != null) {
            scheduledIdleClear.cancel(false);
            scheduledIdleClear = null;
        }     
    }

    private void startIdleClear() throws REBException, RaftException, DAQException {
        startPeriodicTask();
        
    }
    
    /**
     * Puts the sequencers into IDLE_CLEAR state.
     *
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    synchronized void startPeriodicTask() throws REBException, RaftException, DAQException {
        // Todo: Take the idleClear lock, this will be set true until idle clear is done
        synchronized (idleClearLock) {
            boolean wasSet = idleClearLock.getAndSet(true);
            if (wasSet) {
                throw new IllegalStateException("idleClearLock was already set");
            }
        }
        
        sequencers.prepareForIdleClear();

        //Start the periodic task
        idleClearPeriodicTask = idleClearScheduler.scheduleAtFixedRate(() -> idleClear(), 0, sequencerConfig.getIdleClearPeriod().toMillis(), TimeUnit.MILLISECONDS, "idleClearTask", new IdleClearPeriodicTaskExceptionHandler());        
    }
    
    /**
     * Actually perform the idle clear operation
     */
    private void idleClear() {
        synchronized (idleClearLock) {
            boolean wasSet = idleClearLock.get();
            if (!wasSet) {
                return;
            }
        }
        try (AutoCloseableReentrantLock lock = SequencerProc.lockSequencer(sequencerConfig.getIdleClearPeriod().toMillis()/2, TimeUnit.MILLISECONDS)) {
            LOG.log(Level.FINE, "Periodic idleClear");
        
            try {
                sequencers.doIdleClear();
            } catch (DAQException | REBException | RaftException x) {
                LOG.log(Level.SEVERE, "Error during idle clear", x);
            }
        } catch (AutoCloseableReentrantLock.AutoCloseableReentrantLockException e) {
            LOG.log(Level.INFO, "Periodic idleClear skipped: {0}", e.getMessage());            
        }
    }

    /**
     * Stops the sequencers, which must be in IDLE_CLEAR mode.
     * This method does not return until the idle clear is complete.
     *
     * @param stateOnFinish The state to go to, or null to leave the state
     * unchanged.
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    synchronized void endIdleClear(FocalPlaneState stateOnFinish) throws REBException, RaftException, DAQException {
        synchronized (idleClearLock) {
            boolean wasSet = idleClearLock.getAndSet(false);
            if (!wasSet) {
                throw new IllegalStateException("idleClearLock was already clear");
            }
            
            cancelPeriodicTask();
            // This method waits for the sequencers to stop
            sequencers.terminateIdleClear(stateOnFinish);
            // Restore any sequencer parameters which were changed
            // Release the idle clear lock
        }
    }
    // The following methods are called from LSE71Commands on the action thread.
    /**
     * Called to indicate some other command wants to start and needs to wait for idle clear to finish
     */
    void commandedEndIdleClear() throws REBException, RaftException, DAQException {
        synchronized (idleClearLock) {
            if (idleClearLock.get()) {
                endIdleClear(null);
            }
        }
    }
    

    void commandedStartIdleClear() throws REBException, RaftException, DAQException {
        synchronized (idleClearLock) {
            if (!idleClearLock.get()) {
                cancelPauseAndRestart();
            }
        }
    }

    /**
     * Called to pause idle clear from LSE71 commands
     */
    void commandedPauseIdleClear(Duration pause) throws REBException, DAQException, RaftException {
        pause(pause);
    }

    /**
     * Used for suppressing log messages when in idle clear state
     * @return <code>true</code>If message level should be reduced
     */
    boolean isIdleClearActive() {
        return agentStateService != null && agentStateService.isInState(FocalPlaneState.IDLE_CLEAR);
    }
    
    private class IdleClearPeriodicTaskExceptionHandler extends PeriodicTaskExceptionHandler {

        public IdleClearPeriodicTaskExceptionHandler() {
            setSkipOverdueExecutions(true);
        }

        @Override
        public void onSkippedExecutions(PeriodicTask task, int nSkipped) {
            LOG.log(Level.INFO, "Overdue periodic idleClear skipped (n={0})", nSkipped);            
        }
    }

    // Just fot testing
    PeriodicTask getIdleClearPeriodicTask() {
        return idleClearPeriodicTask;
    }
    
    
}
