package org.lsst.ccs.subsystem.focalplane;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.StateBundle;

import org.lsst.ccs.daq.guider.ClearParameters;
import org.lsst.ccs.daq.guider.Config;
import org.lsst.ccs.daq.guider.ROICommon;
import org.lsst.ccs.daq.guider.ROICommonExtended;
import org.lsst.ccs.daq.guider.ROISpec;
import org.lsst.ccs.daq.guider.SeriesStatus;
import org.lsst.ccs.daq.guider.Status;
import org.lsst.ccs.daq.guider.Status.State;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Guider;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.GuidingState;
import org.lsst.ccs.utilities.location.SensorLocationSet;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Main interface for handling guiding in focal-plane
 * @author tonyj
 */
public final class Guiding {

    private static final Logger LOG = Logger.getLogger(Guiding.class.getName());
    private static final Map<State, GuidingState> stateMap = Map.of(
        State.IDLECLEARING, GuidingState.IDLECLEARING,
        State.SLEEPING, GuidingState.SLEEPING,
        State.ERROR, GuidingState.ERROR,
        State.PAUSED, GuidingState.PAUSED,
        State.RUNNING, GuidingState.RUNNING,
        State.UNDEFINED, GuidingState.UNDEFINED,
        State.CLEARING, GuidingState.CLEARING,
        State.IDLEPAUSE, GuidingState.IDLEPAUSE);

    private final SequencerConfig sequencerConfig;
    private final AgentStateService agentStateService;
    private final Scheduler pauseTimeoutScheduler;          
    private ROISpec roiSpec;
    private final static SensorLocationSet NO_SENSORS = new SensorLocationSet();
    private ScheduledFuture<Object> scheduledPauseTimeout;
    private CompletableFuture<Void> readoutActive;

    public Guiding(SequencerConfig config, AgentStateService agentStateService, Scheduler pauseTimeoutScheduler) {
        this.sequencerConfig = config;
        this.agentStateService = agentStateService;
        this.pauseTimeoutScheduler = pauseTimeoutScheduler;

        boolean hasGuiderPartition = sequencerConfig.hasGuiderPartition();
        if (hasGuiderPartition) {
            try {
                State out = sequencerConfig.getGuider().series().getStatus().getOut();
                updateState(out);
                if (out != State.IDLECLEARING && out != State.SLEEPING) {
                    stopGuiding();
                }
            } catch (DAQException x) {
                LOG.log(Level.SEVERE, "Error initializing guider state", x);
                agentStateService.updateAgentState(GuidingState.UNDEFINED);
            }
        } else {
            agentStateService.updateAgentState(GuidingState.UNDEFINED);
        }
    }

    private void updateState(Status.State state) {
        LOG.log(Level.INFO, "Initial guiding state {0}", state);
        agentStateService.updateAgentState(stateMap.get(state));
    }

    /**
     * Called when focal-plane receives initGuiders command
     * @param roiSpec
     * @throws DAQException
     */
    void initGuiders(String roiSpec) throws DAQException {
        this.roiSpec = ROISpec.parse(roiSpec);
        Guider guider = sequencerConfig.getGuider();
        final SeriesStatus series = guider.series();
        Status.State out = series.getStatus().getOut();
        switch (out) {
            case SLEEPING -> guider.wake(sequencerConfig.getGuiderIdleClearParameters());
            case ERROR -> // TODO : Keep this long-term? (LSSTCCSRAFTS-779)
                recoverFromErrorBeforeInitGuiders(guider);
            case PAUSED -> 
            // We want to allow two or more initGuiders in a row (LSSTCCSRAFTS-781)
            // Per Gregg, the DAQ now accepts start command in PAUSED state, so no need to take any special action'
            LOG.log(Level.WARNING, "Accepting initGuiders when guiders already in PAUSED state");
        }
        ROICommonExtended rce = new ROICommonExtended(this.roiSpec.getCommon(), sequencerConfig.getExtendedROIParameters());
        Status status = guider.start(rce, this.roiSpec.getLocations());
        checkStatusAfterCommand(status, GuidingState.PAUSED);
        schedulePauseTimeout();
    }

    private void schedulePauseTimeout() {
        long guiderPauseTimeout = sequencerConfig.getGuiderPauseTimeout();
        if (guiderPauseTimeout > 0) {
            scheduledPauseTimeout = pauseTimeoutScheduler.schedule(() -> {
                synchronized (agentStateService.getStateLock()) {
                    scheduledPauseTimeout = null;
                    if (agentStateService.isInState(GuidingState.PAUSED)) {
                        LOG.log(Level.WARNING, "Guider pause timeout triggered");
                        stopGuiding();
                    }
                }
                return null;
            }, guiderPauseTimeout, TimeUnit.MILLISECONDS);
        }
    }
    
    private void cancelPauseTimeout() {
        ScheduledFuture<?> timeout = scheduledPauseTimeout;
        if (timeout != null) {
            scheduledPauseTimeout = null;
            timeout.cancel(false);
        }
    }


    /**
     * Validate an roispec
     * @param roiSpec The roispec to validate
     * @return The estimated time between stamps, used to estimate when the guiding needs to be stopped
     * @throws DAQException
     */
    int validate(String roiSpec) throws DAQException {
        ROISpec spec = ROISpec.parse(roiSpec);
        Guider guider = sequencerConfig.getGuider();
        spec.sanityCheck(guider.getConfiguredLocations());
        spec.sanityCheckWithSensorLocations(sequencerConfig.getGuiderLocations());
        ROICommon common = spec.getCommon();
        guider.validate(common, spec.getLocations());
        //  Per Gregg: Sequencer_time_in_ns = 32751760 + 490700*RoiRows + 1030*RoiCols*RoiRows
        int rows = common.getRows();
        int cols = common.getCols();
        // TODO: Check/Update this
        int ns = 32751760 + 490700*rows + 1030*cols*rows;
        return common.getIntegrationTimeMillis() + ns/1_000_000;
    }

    /**
     * Resume guiding using the previously set ROIs
     * @param imageName The current imageName
     * @throws DAQException
     */
    void resumeGuiding(String imageName) throws DAQException {
        cancelPauseTimeout();
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.resume(imageName);
            checkStatusAfterCommand(status, GuidingState.RUNNING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    /**
     * Pause guiding
     * @return The number of stamps accumulated since resumeGuiding
     * @throws DAQException
     */
    int pauseGuiding() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.pause();
            checkStatusAfterCommand(status, GuidingState.PAUSED);
            schedulePauseTimeout();            
            int stamps = guider.series().getSeries().getStamps();
            return stamps;
        } catch (DAQException x) {
            State in = guider.series().getStatus().getIn();
            if (in == State.ERROR) {
                recoverFromErrorBeforePause(guider);
                return 0;
            }
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }
    
    private void recoverFromErrorBeforePause(Guider guider) throws DAQException {
        try {
            // At Gregg's request we also want to log the daq status in this case
            SeriesStatus series = guider.series();
            LOG.log(Level.WARNING, "Attempting to recover from ERROR before pause, transitioning to IDLECLEARING, previous state {0}", series);
            Status status = guider.stop();
            checkStatusAfterCommand(status, GuidingState.IDLECLEARING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));            
            throw x;
        }
    }
    
    private void recoverFromErrorBeforeInitGuiders(Guider guider) throws DAQException {
        try {
            // At Gregg's request we also want to log the daq status in this case
            SeriesStatus series = guider.series();
            LOG.log(Level.WARNING, "Attempting to recover from ERROR before initGuiders, transitioning to IDLECLEARING, previous state {0}", series);
            Status status = guider.stop();
            checkStatusAfterCommand(status, GuidingState.IDLECLEARING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }
    
    /**
     * Called to pause guider clearing during readout IF no active guiding happening.
     */
    void pauseDuringReadout() throws DAQException {
        if (this.isROISet()) {
            readoutActive = new CompletableFuture<>();
        } else {
            Guider guider = sequencerConfig.getGuider();
            try {
                Status status = guider.pause();
                checkStatusAfterCommand(status, GuidingState.IDLEPAUSE);
            } catch (DAQException x) {
                State out = guider.series().getStatus().getOut();
                agentStateService.updateAgentState(stateMap.get(out));
                throw x;
            }
        }
    }
    
    /**
     * Called to resume guider clearing during readout IF no active guiding happening.
     */
    void resumeAfterReadout() throws DAQException {
        if (this.isROISet()) {
            CompletableFuture<Void> localReadout = readoutActive;
            if (localReadout != null) { 
                localReadout.complete(null);
                readoutActive = null;
            }
        } else {
            Guider guider = sequencerConfig.getGuider();
            try {
                Status status = guider.resume();
                checkStatusAfterCommand(status, GuidingState.IDLECLEARING);
            } catch (DAQException x) {
                State out = guider.series().getStatus().getOut();
                agentStateService.updateAgentState(stateMap.get(out));
                throw x;
            }    
        }
    }

    boolean isROISet() {
        return roiSpec != null;
    }

    void stopGuidingOnceReadoutComplete() throws DAQException {
        CompletableFuture<Void> localReadout = readoutActive;
        if (localReadout  != null) {
            LOG.log(Level.INFO, "Scheduling stopGuiding after readout complete");
            localReadout.thenRun(() -> {
                try {
                    stopGuiding();
                    LOG.log(Level.INFO, "Scheduled stopGuiding done");
                } catch (DAQException x) {
                    LOG.log(Level.SEVERE, "Scheduled stopGuiding failed", x);
                }
            });
        } else {
            stopGuiding();
        }
    }
    
    void stopGuiding() throws DAQException {
        cancelPauseTimeout();
        roiSpec = null;
        Guider guider = sequencerConfig.getGuider();
        State out = guider.series().getStatus().getOut();
        if (out != State.SLEEPING && out != State.UNDEFINED && out != State.IDLECLEARING) {
            try {
                Status status = guider.stop();
                checkStatusAfterCommand(status, GuidingState.IDLECLEARING);
            } catch (DAQException x) {
                out = guider.series().getStatus().getOut();
                agentStateService.updateAgentState(stateMap.get(out));
                throw x;
            }
        } else {
            agentStateService.updateAgentState(stateMap.get(out));
        }
    }

    SensorLocationSet getSensors() {
        return roiSpec == null ? NO_SENSORS : roiSpec.getSensorLocations();
    }

    void sleepGuider() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.sleep();
            checkStatusAfterCommand(status, GuidingState.SLEEPING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    void wakeGuider() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.wake(sequencerConfig.getGuiderIdleClearParameters());
            checkStatusAfterCommand(status, GuidingState.IDLECLEARING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    void clearGuider(ClearParameters clearParameters) throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.clear(clearParameters);
            checkStatusAfterCommand(status, GuidingState.CLEARING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    SeriesStatus series() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        return guider.series();
    }

    Config config() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        return guider.config();
    }

    private void checkCurrentStatus(GuidingState expectedState) {
        final StateBundle currentState = agentStateService.getState();
        if (!currentState.isInState(expectedState)) {
            throw new RuntimeException("Guider in unexpected state, expected="+expectedState+" actual="+currentState.getState(GuidingState.class));            
        }
    }
    
    private void checkStatusAfterCommand(Status status, GuidingState expectedState) {
        GuidingState newState = mapStateToGuidingState(status.getOut());
        agentStateService.updateAgentState(newState);
        if (newState != expectedState) {
            throw new RuntimeException("Guider entered unexpected state, expected="+expectedState+" actual="+newState);
        }
    }

    private GuidingState mapStateToGuidingState(State state) {
        return stateMap.getOrDefault(state, GuidingState.UNDEFINED);
    }

}
