package org.lsst.ccs.subsystem.focalplane;

import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.daq.guider.ClearParameters;
import org.lsst.ccs.daq.guider.Config;
import org.lsst.ccs.daq.guider.ROICommon;
import org.lsst.ccs.daq.guider.ROICommonExtended;
import org.lsst.ccs.daq.guider.ROISpec;
import org.lsst.ccs.daq.guider.SeriesStatus;
import org.lsst.ccs.daq.guider.Status;
import org.lsst.ccs.daq.guider.Status.State;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Guider;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.GuidingState;
import org.lsst.ccs.utilities.location.SensorLocationSet;

/**
 * Main interface for handling guiding in focal-plane
 * @author tonyj
 */
public class Guiding {

    private static final Logger LOG = Logger.getLogger(Guiding.class.getName());
    private static Map<State, GuidingState> stateMap = Map.of(
        State.STOPPED, GuidingState.STOPPED,
        State.OFF, GuidingState.OFF,
        State.ERROR, GuidingState.ERROR,
        State.PAUSED, GuidingState.PAUSED,
        State.RUNNING, GuidingState.RUNNING,
        State.UNDEFINED, GuidingState.UNDEFINED,
        State.CLEARING, GuidingState.CLEARING);

    private final SequencerConfig sequencerConfig;
    private final AgentStateService agentStateService;
    private ROISpec roiSpec;
    private final static SensorLocationSet NO_SENSORS = new SensorLocationSet();

    public Guiding(SequencerConfig config, AgentStateService agentStateService) {
        this.sequencerConfig = config;
        this.agentStateService = agentStateService;
        boolean hasGuiderPartition = sequencerConfig.hasGuiderPartition();
        if (hasGuiderPartition) {
            try {
                State out = sequencerConfig.getGuider().series().getStatus().getOut();
                updateState(out);
                if (out != State.STOPPED && out != State.OFF) {
                    stopGuiding();
                }
            } catch (DAQException x) {
                LOG.log(Level.SEVERE, "Error initializing guider state", x);
                agentStateService.updateAgentState(GuidingState.UNDEFINED);
            }
        } else {
            agentStateService.updateAgentState(GuidingState.UNDEFINED);
        }
    }

    private void updateState(Status.State state) {
        LOG.log(Level.INFO, "Initial guding state {0}", state);
        agentStateService.updateAgentState(stateMap.get(state));
    }

    /**
     * Called when focal-plane receives initGuiders command
     * @param roiSpec
     * @throws DAQException
     */
    void initGuiders(String roiSpec) throws DAQException {
        this.roiSpec = ROISpec.parse(roiSpec);
        Guider guider = sequencerConfig.getGuider();
        final SeriesStatus series = guider.series();
        Status.State out = series.getStatus().getOut();
        if (out == Status.State.OFF) {
            guider.wake(sequencerConfig.getGuiderIdleClearParameters());
        } else if (out == Status.State.ERROR) {
            // TODO : Keep this long-term? (LSSTCCSRAFTS-779)
            recoverFromErrorBeforeInitGuiders(guider);
        } else if (out == Status.State.PAUSED) {
            LOG.log(Level.WARNING, "Accepting initGuiders when guiders already in PAUSED state");
            // We want to allow two or more initGuiders in a row (LSSTCCSRAFTS-781)
            Status status = guider.stop();
            checkStatus(status, GuidingState.STOPPED);
        }
        ROICommonExtended rce = new ROICommonExtended(this.roiSpec.getCommon(), sequencerConfig.getExtendedROIParameters());
        Status status = guider.start(rce, this.roiSpec.getLocations());
        checkStatus(status, GuidingState.PAUSED);
    }

    /**
     * Validate an roispec
     * @param roiSpec The roispec to validate
     * @return The estimated time between stamps, used to estimate when the guiding needs to be stopped
     * @throws DAQException
     */
    int validate(String roiSpec) throws DAQException {
        ROISpec spec = ROISpec.parse(roiSpec);
        Guider guider = sequencerConfig.getGuider();
        spec.sanityCheck(guider.getConfiguredLocations());
        spec.sanityCheckWithSensorLocations(sequencerConfig.getGuiderLocations());
        ROICommon common = spec.getCommon();
        guider.validate(common, spec.getLocations());
        //  Per Gregg: Sequencer_time_in_ns = 32751760 + 490700*RoiRows + 1030*RoiCols*RoiRows
        int rows = common.getRows();
        int cols = common.getCols();
        // TODO: Check/Update this
        int ns = 32751760 + 490700*rows + 1030*cols*rows;
        return common.getIntegrationTimeMillis() + ns/1_000_000;
    }

    /**
     * Resume guiding using the previously set ROIs
     * @param imageName The current imageName
     * @throws DAQException
     */
    void resumeGuiding(String imageName) throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.resume(imageName);
            checkStatus(status, GuidingState.RUNNING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    /**
     * Pause guiding
     * @return The number of stamps accumulated since resumeGuiding
     * @throws DAQException
     */
    int pauseGuiding() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.pause();
            checkStatus(status, GuidingState.PAUSED);
            int stamps = guider.series().getSeries().getStamps();
            return stamps;
        } catch (DAQException x) {
            State in = guider.series().getStatus().getIn();
            if (in == State.ERROR) {
                recoverFromErrorBeforePause(guider);
                return 0;
            }
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }
    
    private void recoverFromErrorBeforePause(Guider guider) throws DAQException {
        try {
            LOG.log(Level.WARNING, "Attempting to recover from ERROR before pause, transitioning to STOPPED");
            Status status = guider.stop();
            checkStatus(status, GuidingState.STOPPED);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));            
            throw x;
        }
    }
    
    private void recoverFromErrorBeforeInitGuiders(Guider guider) throws DAQException {
        try {
            LOG.log(Level.WARNING, "Attempting to recover from ERROR before initGuiders, transitioning to STOPPED");
            Status status = guider.stop();
            checkStatus(status, GuidingState.STOPPED);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();                
            agentStateService.updateAgentState(stateMap.get(out));            
            throw x;
        }        
    }

    boolean isROISet() {
        return roiSpec != null;
    }

    void stopGuiding() throws DAQException {
        roiSpec = null;
        Guider guider = sequencerConfig.getGuider();
        State out = guider.series().getStatus().getOut();
        if (out != State.OFF && out != State.UNDEFINED && out != State.STOPPED) {
            try {
                Status status = guider.stop();
                checkStatus(status, GuidingState.STOPPED);
            } catch (DAQException x) {
                out = guider.series().getStatus().getOut();
                agentStateService.updateAgentState(stateMap.get(out));
                throw x;
            }
        } else {
            agentStateService.updateAgentState(stateMap.get(out));
        }
    }

    SensorLocationSet getSensors() {
        return roiSpec == null ? NO_SENSORS : roiSpec.getSensorLocations();
    }

    void sleepGuider() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.sleep();
            checkStatus(status, GuidingState.OFF);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    void wakeGuider() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.wake(sequencerConfig.getGuiderIdleClearParameters());
            checkStatus(status, GuidingState.STOPPED);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    void clearGuider(ClearParameters clearParameters) throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        try {
            Status status = guider.clear(clearParameters);
            checkStatus(status, GuidingState.CLEARING);
        } catch (DAQException x) {
            State out = guider.series().getStatus().getOut();
            agentStateService.updateAgentState(stateMap.get(out));
            throw x;
        }
    }

    SeriesStatus series() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        return guider.series();
    }

    Config config() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        return guider.config();
    }

    private void checkStatus(Status status, GuidingState expectedState) {
        GuidingState newState = mapStateToGuidingState(status.getOut());
        agentStateService.updateAgentState(newState);
        if (newState != expectedState) {
            throw new RuntimeException("Guider entered unexpected state, expected="+expectedState+" actual="+newState);
        }
    }

    private GuidingState mapStateToGuidingState(State state) {
        return stateMap.getOrDefault(state, GuidingState.UNDEFINED);
    }
}
