package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.daq.ims.Camera;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.ReadoutMode;
import org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent;
import org.lsst.ccs.subsystem.focalplane.data.SequencerInfo;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.focalplane.states.SequencerState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Reb.RebType;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Encapsulates the behavior of all Sequencers in the focal plane.
 *
 * @author tonyj
 */
public class Sequencers {

    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());
    private final Lock sequencerLock = new ReentrantLock();
    private volatile Condition sequencerFinished;

    private final Map<Reb, SequencerProc> REBs = new LinkedHashMap<>();
    private final Map<Location, Reb> locationMap = new LinkedHashMap<>();
    private final Map<Location, Map<String, Integer>> sequencerParameterCache = new LinkedHashMap<>();
    private ImageName imageName;
    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig sequencerConfig;
    private Map<SequencerType, ModelAndName> models;
    private Map<String, Integer> opcodeMap;
    private Integer currentOpcode;
    private Boolean supportsMultiMains;
    private Map<Reb.RebType, int[]> registerMap;
    private KeyValueDataList sequencerKeyValueData;
    private String annotation;
    private LocationSet locations;
    private final LocationSet configuredLocations = new LocationSet();
    private final RebDevices devices;

    Runnable testIfShouldRunIdleFlush;
    Runnable testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle;
    private final ExecutorService executor;

    Sequencers(FocalPlaneSubsystem subsys, SequencerConfig sequencer, ExecutorService executor) {
        this.subsys = subsys;
        this.sequencerConfig = sequencer;
        this.executor = executor;
        this.devices = new RebDevices(executor, sequencer, configuredLocations);
        this.testIfShouldRunIdleFlush = () -> {
            if (sequencerConfig.getIdleFlushTimeout() == 0) {
                try {
                    startIdleFlush();
                } catch (DAQException | REBException | RaftException x) {
                    LOG.log(Level.SEVERE, "Error starting IDLE_FLUSH", x);
                    setState(SequencerState.IDLE);
                }
            } else {
                setState(SequencerState.IDLE);
            }
        };
        this.testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle = () -> {
            testIfShouldRunIdleFlush.run();
            setState(FocalPlaneState.QUIESCENT);
        };
    }

    void add(Reb reb, SequencerProc seq) {
        REBs.put(reb, seq);
        Location location = reb.getLocation();
        configuredLocations.add(location);
        locationMap.put(location, reb);
        if (subsys != null && subsys.dataProviderDictionaryService != null) {
            subsys.dataProviderDictionaryService.registerClass(SequencerInfo.class, reb.getFullName());
            subsys.dataProviderDictionaryService.registerClass(ImageReadoutParametersEvent.class, ImageReadoutParametersEvent.EVENT_KEY);
        }
    }

    /**
     * Run the requested number of clears, then go into QUIESCENT state
     *
     * @param nClears The number of clears
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    public void clear(int nClears) throws REBException, RaftException, DAQException {
        clear(nClears, testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle);
    }

    /**
     * Run the requested number of clears, the go into specified state
     *
     * @param nClears The number of clears
     * @param runAfterClear Runnable to run when clear finishes (or
     * <code>null</code>)
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    public void clear(int nClears, Runnable runAfterClear) throws REBException, RaftException, DAQException {
        setMain(sequencerConfig.getClearMain());
        setParameter(sequencerConfig.getClearCountParameter(), nClears);
        sequencerFinished = scheduleEmulatedSequencer(Duration.ofMillis(nClears * 100));
        runSequencers(false, FocalPlaneState.CLEARING, SequencerState.RUNNING, runAfterClear);
    }

    public void startIntegration(ImageName imageName, String annotation, LocationSet locations) throws REBException, RaftException, DAQException {
        if (!sequencerConfig.isRealDAQ()) {
            CCSPlayList playList = sequencerConfig.getCurrentPlaylist();
            if (playList == null) {
                throw new RaftException("Using emulated DAQ but no playlist defined");
            }
            if (!playList.hasNextImage()) {
                throw new RaftException("No images left in playlist " + playList.getName());
            }
        }
        this.imageName = imageName;
        this.annotation = annotation;
        this.locations = locations;
        if (sequencerConfig.useStepAfterIntegrate()) {
            setSubroutinePointer(sequencerConfig.getIntegrateAfterPointer(), sequencerConfig.getReadSubroutine());
        }
        setMain(sequencerConfig.getIntegrateMain());
        if (sequencerConfig.hasEmulatedDAQ()) {
            sequencerFinished = sequencerLock.newCondition();
        }
        runSequencers(false, FocalPlaneState.INTEGRATING, SequencerState.RUNNING, (Runnable) null);
    }

    public void rowShift(int nRows) throws REBException, RaftException, DAQException {
        if (nRows == 0) {
            return;
        }
        if (sequencerConfig.useStepAfterIntegrate()) {
            throw new RaftException("rowShift not suported when useStepAfterIntegrate is true");
        }
        if (sequencerConfig.hasEmulatedDAQ()) {
            throw new RaftException("rowShift not suported with emulatedDAQ");
        }
        stop();
        waitForStop(Duration.ofSeconds(1));
        if (nRows > 0) {
            setMain(sequencerConfig.getRowShiftForwardMain());
        } else {
            setMain(sequencerConfig.getRowShiftReverseMain());
        }
        setParameter(sequencerConfig.getShiftCountParameter(), Math.abs(nRows));
        sequencerFinished = null;
        runSequencers(false, FocalPlaneState.ROW_SHIFT, SequencerState.RUNNING, () -> {
            try {
                setMain(sequencerConfig.getIntegrateMain());
                if (sequencerConfig.hasEmulatedDAQ()) {
                    sequencerFinished = sequencerLock.newCondition();
                }
                runSequencers(false, FocalPlaneState.INTEGRATING, SequencerState.RUNNING, (Runnable) null);
            } catch (REBException | RaftException | DAQException x) {
                LOG.log(Level.SEVERE, "Error restarting INTEGRATE after rowShift", x);
                setState(SequencerState.IDLE);
            }
        });
    }

    public void stop() throws DAQException, REBException {
        if (sequencerConfig.hasEmulatedDAQ()) {
            Condition condition = sequencerFinished;
            if (condition != null) {
                sequencerLock.lock();
                try {
                    condition.signalAll();
                } finally {
                    sequencerLock.unlock();
                }
            }
        } else if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            sequencerConfig.getCamera().startSequencer(Camera.OPCODE_STEP);
        } else {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().sendStop();
                }
            }
        }
    }

    public Image endIntegration(ReadoutMode readout) throws REBException, RaftException, DAQException, IOException {
        LocationSet effectiveLocations = getEffectiveLocations();
        if (sequencerConfig.useStepAfterIntegrate()) {
            // TODO: Experimental code, get rid of copy/paste from runSequencers
            CCSTimeStamp ts = CCSTimeStamp.currentTime();
            // In the case of stepAfterIntegrate we always trigger the DAQ, so in emulated/simulated mode must set playlist
            if (!sequencerConfig.isRealDAQ()) {
                CCSPlayList playList = sequencerConfig.getCurrentPlaylist();
                playList.playNextImage(imageName.toString(), effectiveLocations, Camera.OPCODE_STEP, annotation);
            }
            switch (readout) {

                case TRUE:
//                    if (!sequencerConfig.isRealDAQ()) {
//                        CCSPlayList playList = sequencerConfig.getCurrentPlaylist();
//                        playList.playNextImage(effectiveLocations, annotation);
//                    }
                    ImageMetaData imd = new ImageMetaData(imageName.toString(), sequencerConfig.getDAQFolder(), annotation, Camera.OPCODE_STEP, effectiveLocations);
                    Image image;
                    if (getNRebs() > 0) {
                        image = sequencerConfig.getCamera().triggerImage(imd);
                        Duration delta = Duration.between(ts.getTAIInstant(), image.getMetaData().getTimestamp());
                        LOG.log(Level.INFO, "Delta time between CCS and DAQ={0}", delta);                        
                    } else {
                        image = null;
                    }
                    setState(ts, FocalPlaneState.READING_OUT, SequencerState.RUNNING);
                    sendImageReadoutParametersEvent(effectiveLocations);
                    doInBackground(() -> {
                        try {
                            // No way to tell how long we have to wait for infinite loop,
                            // so wait a somewhat arbitrary 1,000,000 seconds.
                            waitForStop(Duration.ofMillis(1_000_000_000));
                            LOG.info("Sequencers finished");
                            if (subsys != null) {
                                subsys.getImageCoordinatorService().sequencersFinished();
                            }
                            testIfShouldRunIdleFlush.run();
                        } catch (REBException ex) {
                            LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
                        }
                    });
                    return image;
                case PSEUDO:
                    if (getNRebs() > 0) {
                        setSubroutinePointer(sequencerConfig.getIntegrateAfterPointer(), sequencerConfig.getPseudoSubroutine());
                        Instant daqTime = sequencerConfig.getCamera().startSequencer(Camera.OPCODE_STEP);
                        Duration delta = Duration.between(ts.getTAIInstant(), daqTime);
                        LOG.log(Level.INFO, "Delta time between CCS and DAQ={0}", delta);
                    }
                    setState(ts, FocalPlaneState.READING_OUT, SequencerState.RUNNING);
                    sendImageReadoutParametersEvent(effectiveLocations);
                    doInBackground(() -> {
                        try {
                            // No way to tell how long we have to wait for infinite loop,
                            // so wait a somewhat arbitrary 1,000,000 seconds.
                            waitForStop(Duration.ofMillis(1_000_000_000));
                            LOG.info("Sequencers finished");
                            testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle.run();
                        } catch (REBException ex) {
                            LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
                        }
                    });
                    return null;
                default:
                case FALSE:
                    if (getNRebs() > 0) {
                        setSubroutinePointer(sequencerConfig.getIntegrateAfterPointer(), sequencerConfig.getNoOpSubroutine());
                        sequencerConfig.getCamera().startSequencer(Camera.OPCODE_STEP);
                    }
                    waitForStop(Duration.ofSeconds(5));
                    testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle.run();
                    // Nothing else to do
                    return null;
            }
        } else { // No StepAfterIntegrate
            stop();
            waitForStop(Duration.ofSeconds(1));
            switch (readout) {
                case TRUE:
                    // When not using stepAfterIntegrate we only trigger the DAQ on a real readout
                    setMain(sequencerConfig.getReadMain());
                    if (!sequencerConfig.isRealDAQ()) {
                        CCSPlayList playList = sequencerConfig.getCurrentPlaylist();
                        playList.playNextImage(imageName.toString(), effectiveLocations, currentOpcode, annotation);
                    }
                    sequencerFinished = scheduleEmulatedSequencer(Duration.ofSeconds(2));
                    return runSequencers(true, FocalPlaneState.READING_OUT, SequencerState.RUNNING, () -> {
                        if (subsys != null) {
                            subsys.getImageCoordinatorService().sequencersFinished();
                        }
                        testIfShouldRunIdleFlush.run();
                    });
                case PSEUDO:
                    setMain(sequencerConfig.getPseudoReadMain());
                    sequencerFinished = scheduleEmulatedSequencer(Duration.ofSeconds(2));
                    runSequencers(false, FocalPlaneState.READING_OUT, SequencerState.RUNNING, testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle);
                    return null;
                default:
                case FALSE:
                    sequencerFinished = null;
                    testIfShouldRunIdleFlushAndSetFocalPlaneStateToIdle.run();
                    // Nothing else to do
                    return null;
            }
        }
    }

    private Condition scheduleEmulatedSequencer(Duration when) {
        if (sequencerConfig.hasEmulatedDAQ()) {
            Condition condition = sequencerLock.newCondition();
            subsys.getScheduler().schedule(() -> {
                sequencerLock.lock();
                try {
                    condition.signalAll();
                } finally {
                    sequencerLock.unlock();
                }
            }, when.toMillis(), TimeUnit.MILLISECONDS);
            return condition;
        } else {
            return null;
        }
    }

    /**
     * Puts the sequencers into IDLE_FLUSH state.
     *
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    public void startIdleFlush() throws REBException, RaftException, DAQException {
        // TODO: Does this work with emulated DAQ?
        setMain(sequencerConfig.getIdleFlushMain());
        if (sequencerConfig.hasEmulatedDAQ()) {
            sequencerFinished = sequencerLock.newCondition();
        }
        // Since IDLE_FLUSH will never finish spontaneously we do not set a completion state and leave it up to whoever
        // stops the sequencer to decide what state it should transition to.
        // NOTE: We do not force focal-plane to QUIESCENT, in case it is currently in IDLE_FLUSH
        runSequencers(false, () -> setState(SequencerState.IDLE_FLUSH));
    }

    /**
     * Stops the sequencers, which is assumed to be in IDLE_FLUSH mode.
     *
     * @param stateOnFinish The state to go to, or null to leave the state
     * unchanged.
     * @throws REBException
     * @throws RaftException
     * @throws DAQException
     */
    public void endIdleFlush(FocalPlaneState stateOnFinish) throws REBException, RaftException, DAQException {
        // TODO: Does this work with emulated DAQ?
        stop();
        waitForStop(Duration.ofSeconds(1));
        if (stateOnFinish != null) {
            setState(stateOnFinish, SequencerState.IDLE);
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        // If no sequencers were loaded, e.g. if there are no science sensors defined, opcodeMap
        // will be null. We need to support this case for operating TS8 and AuxTel in guider mode
        if (opcodeMap != null) {
            currentOpcode = opcodeMap.get(mainName);
            if (currentOpcode == null) {
                throw new RaftException("Invalid main: " + mainName);
            }
            if ((!Objects.equals(Boolean.TRUE, supportsMultiMains)) || subsys == null) {
                for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                    if (isRebOnlineAndValid(entry.getKey())) {
                        entry.getValue().setStart(mainName);
                    }
                }
            }
        } else {
            currentOpcode = null;
        }
    }

    public void setSubroutinePointer(String pointerName, String subroutineName) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            if (isRebOnlineAndValid(entry.getKey())) {
                final SequencerProc sequenceProc = entry.getValue();
                final Integer subroutineAddress = sequenceProc.getSubroutineMap().get(subroutineName);
                if (subroutineAddress == null) {
                    throw new RaftException("Subroutine " + subroutineName + " not found");
                }
                sequenceProc.setParameter(pointerName, subroutineAddress);
            }
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            if (isRebOnlineAndValid(entry.getKey())) {
                entry.getValue().setParameter(parameter, value);
            }
        }
    }

    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, stateWhileRunning, SequencerState.RUNNING);
    }

    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning, SequencerState sequencerStateWhileRunning) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, stateWhileRunning, sequencerStateWhileRunning, (FocalPlaneState) null);
    }

    private Image runSequencers(boolean acquire, Runnable runOnStart) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, runOnStart, null);
    }

    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning, SequencerState sequencerStateWhileRunning, FocalPlaneState stateOnCompletion) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, stateWhileRunning, sequencerStateWhileRunning, stateOnCompletion == null ? null : () -> setState(stateOnCompletion, SequencerState.IDLE));
    }

    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning, SequencerState sequencerStateWhileRunning, Runnable runOnFinish) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, () -> setState(stateWhileRunning, sequencerStateWhileRunning), runOnFinish);
    }

    private Image runSequencers(boolean acquire, Runnable runOnStart, Runnable runOnFinish) throws REBException, RaftException, DAQException {
        LocationSet effectiveLocations = getEffectiveLocations();
        Image image = null;
        CCSTimeStamp ts = CCSTimeStamp.currentTime();
        if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            if (currentOpcode == null) {
                throw new REBException("opcode not set");
            }
            LOG.log(Level.INFO, "Starting sequencers with opcode={0}", currentOpcode);
            if (acquire) {
                ImageMetaData imd = new ImageMetaData(imageName.toString(), sequencerConfig.getDAQFolder(), annotation, currentOpcode, effectiveLocations);
                if (getNRebs() > 0) {
                    image = sequencerConfig.getCamera().triggerImage(imd);
                    Duration delta = Duration.between(ts.getTAIInstant(), image.getMetaData().getTimestamp());
                    LOG.log(Level.INFO, "Delta time between CCS and DAQ={0}", delta);
                }
                sendImageReadoutParametersEvent(effectiveLocations);
            } else {
                Instant daqTime = sequencerConfig.getCamera().startSequencer(currentOpcode);
                Duration delta = Duration.between(ts.getTAIInstant(), daqTime);
                LOG.log(Level.INFO, "Delta time between CCS and DAQ={0}", delta);
            }
        } else if (subsys != null) {
            int oldOpcode = 1; // Opcode to be used with pre-multi-main firmware
            if (acquire) {
                ImageMetaData imd = new ImageMetaData(imageName.toString(), sequencerConfig.getDAQFolder(), annotation, oldOpcode, effectiveLocations);
                image = sequencerConfig.getCamera().triggerImage(imd);
                sendImageReadoutParametersEvent(effectiveLocations);
            } else {
                sequencerConfig.getCamera().startSequencer(oldOpcode);
            }
        } else {
            // Used for simulation only
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().startSequencer();
                }
            }
            if (acquire) {
                sendImageReadoutParametersEvent(effectiveLocations);
            }
        }
        //TODO: What happens if above fails, state should not be left unchanged.
        runOnStart.run();
        if (runOnFinish != null) {
            if (sequencerConfig.useParallelRegisters()) {
                devices.whenSequencersStopped(() -> {
                    LOG.info("Sequencers finished");
                    runOnFinish.run();
                });
            } else {
                doInBackground(() -> {
                    try {
                        // No way to tell how long we have to wait for infinite loop,
                        // so wait a somewhat arbitrary 1,000,000 seconds.
                        waitForStop(Duration.ofMillis(1_000_000_000));
                        LOG.info("Sequencers finished");
                        runOnFinish.run();
                    } catch (REBException ex) {
                        LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
                    }
                });

            }

        }
        return image;
    }

    private void sendImageReadoutParametersEvent(LocationSet effectiveLocations) throws RaftException, REBException {
        ImageReadoutParametersEvent irpe
                = new ImageReadoutParametersEvent(imageName, annotation, effectiveLocations, currentOpcode, sequencerConfig.getDAQFolder());
        for (Location location : effectiveLocations) {
            Reb reb = locationMap.get(location);
            // LSSTCCSRAFTS-580 Use the sequencer parameter cache rather than actually reading
            // the parameters from the sequencer, to avoid 75*10 register reads.
            Map<String, Integer> cache = sequencerParameterCache.get(location);
            int preCols = cache.get("PreCols");
            int underCols = cache.get("UnderCols");
            int readCols = cache.get("ReadCols");
            int postCols = cache.get("PostCols");
            int readCols2 = cache.get("ReadCols2");
            int overCols = cache.get("OverCols");
            int preRows = cache.get("PreRows");
            int readRows = cache.get("ReadRows");
            int postRows = cache.get("PostRows");
            int overRows = cache.get("OverRows");
            irpe.setParameters(location, reb.getCCDType(), preCols, underCols, readCols, postCols, readCols2, overCols,
                    preRows, readRows, postRows, overRows);
        }
        if (subsys != null) {
            subsys.sendEvent(ImageReadoutParametersEvent.EVENT_KEY, irpe);
        }
    }

    public void waitForStop(Duration timeout) throws REBException {
        if (sequencerConfig.hasEmulatedDAQ()) {
            try {
                Condition condition = sequencerFinished;
                if (condition != null) {
                    sequencerLock.lock();
                    try {
                        condition.await((int) timeout.toMillis(), TimeUnit.MILLISECONDS);
                    } finally {
                        sequencerLock.unlock();
                    }
                }
            } catch (InterruptedException ex) {
                throw new REBException("Interrupt while waiting for sequencer");
            }
        } else {
            long start = System.nanoTime();
            if (sequencerConfig.useParallelRegisters()) {
                devices.waitSequencersStopped(timeout);
                // TODO: Also check for errors?
                // TODO: What about timeout?
                LOG.log(Level.INFO, () -> String.format("Waiting for sequencers took %,dns (timeout %s)", System.nanoTime() - start, timeout));
            } else {
                // We assume that we only need to wait for one REB for each sequencer type. LSSTCCSRAFTS-594.
                Set<String> alreadyChecked = new HashSet();
                for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                    ModelAndName modelAndNameForREB = getModelAndNameForREB(models, entry.getKey());
                    // Note, the real sequencers are synchronous, the simulated sequencers are not
                    if (!alreadyChecked.contains(modelAndNameForREB.getName()) || subsys == null || subsys.isSimulationMode()) {
                        if (isRebOnlineAndValid(entry.getKey())) {
                            SequencerProc seq = entry.getValue();
                            // NOTE: waitDone throws a REBException it if times out
                            // We should go into FAULT state and abort the readout if it does
                            seq.waitDone((int) timeout.toMillis());
                            int errorAddr = seq.getErrorAddr();
                            if (errorAddr != -1) {
                                LOG.log(Level.SEVERE, "REB error register set to {0} for {1}", new Object[]{errorAddr, entry.getKey()});
                                //TODO: We should go into fault state here
                            }
                            alreadyChecked.add(modelAndNameForREB.getName());
                        }
                    }
                }
                LOG.log(Level.INFO, () -> String.format("Waiting for %d/%d sequencers took %,dns (timeout %s)", alreadyChecked.size(), REBs.size(), System.nanoTime() - start, timeout));
            }
        }
    }

    /**
     * Fail fast if the sequencers loaded do not meet our minimal requirements
     *
     * @param seq The sequencer to test
     * @throws RaftException If any of the required mains or parameters are
     * missing
     */
    private static void sanityCheck(List<String> requiredMains, List<String> requiredParameters, FPGA2Model model) throws RaftException, REBException {
        Map<String, Integer> mainMap = model.getMainAddresses();
        for (String main : requiredMains) {
            if (mainMap.get(main) == null) {
                throw new RaftException("Sequencer failed sanity check, missing main " + main);
            }
        }
        List<FPGA2Model.PointerInfo> pointers = model.getPointers();
        for (String parameter : requiredParameters) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (pointer.getName().equals(parameter)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
            }
        }
    }

    /**
     * Check that all required meta-data parameters are present in the
     * sequencer.
     *
     * @param metaData A list of required parameters
     * @param model The sequencer model to check
     * @return The list of addresses of the meta-data registers
     * @throws RaftException If one or more meta-data parameters are missing or
     * invalid
     */
    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map<String, PointerInfo> pointerMap = model.getPointerMap();
        int k = 0;
        for (String metaName : metaDataNames) {
            PointerInfo pi = pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION:
                        registers[k++] = pi.getAddress();
                        break;
                    case REPEAT_SUBROUTINE:
                        registers[k++] = pi.getAddress();
                        break;
                    default:
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());

                }
            } else {
                throw new RaftException("Required parameter not defined " + metaName);
            }
        }
        LOG.log(Level.INFO, "Meta-data registers set {0}", Arrays.toString(registers));
        return registers;
    }

    static FPGA2Model validate(List<String> requiredMains, List<String> requiredParameters, String sequencerFile) throws RaftException, REBException, IOException {
        InputStream input = BootstrapResourceUtils.getBootstrapResource(sequencerFile);
        if (input == null) {
            throw new IOException("Invalid sequencer file name " + sequencerFile);
        }
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        FPGA2Model compiledModel = builder.compileFile(input);

        sanityCheck(requiredMains, requiredParameters, compiledModel);
        return compiledModel;
    }

    /**
     * Called to load the configured sequencers into all registered
     * SequencerProcs.
     *
     * @throws RaftException
     * @throws REBException
     */
    void load() throws RaftException, REBException, DAQException {
        Map<SequencerType, ModelAndName> localModels = new LinkedHashMap<>();
        Map<Reb.RebType, int[]> localRegisterMap = new LinkedHashMap<>();
        opcodeMap = null;

        Map<SequencerType, String> sequencers = sequencerConfig.getSequencers();
        for (Map.Entry<SequencerType, String> sequencer : sequencers.entrySet()) {
            try {
                FPGA2Model model = validate(sequencerConfig.getRequiredMains(), sequencerConfig.getRequiredParameters(), sequencer.getValue());
                localModels.put(sequencer.getKey(), new ModelAndName(model, sequencer.getValue()));
            } catch (IOException x) {
                throw new RaftException("Error reading sequencer " + sequencer.getValue(), x);
            }
        }
        SequencerInfo data;
        this.sequencerKeyValueData = new KeyValueDataList();

        // Now loop over the REBS, and load the appropriate sequencer for each one
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            Reb reb = entry.getKey();
            SequencerProc seq = entry.getValue();
            ModelAndName modelAndName = getModelAndNameForREB(localModels, reb);
            int[] registers = computeMetaDataRegisters(sequencerConfig.getMetaDataRegisters(), modelAndName.getModel());
            if (localRegisterMap.get(reb.getRebType()) == null) {
                localRegisterMap.put(reb.getRebType(), registers);
            } else if (!Arrays.equals(localRegisterMap.get(reb.getRebType()), registers)) {
                throw new RaftException("Sequencers have inconsistent meta-data register sets for type " + reb.getRebType());
            }
            if (opcodeMap == null) {
                opcodeMap = modelAndName.getModel().getMainOpcodes();
            } else if (!opcodeMap.equals(modelAndName.getModel().getMainOpcodes())) {
                throw new RaftException("Sequencers have inconsistent main opcodes\n" + opcodeMap + "\n" + modelAndName.getModel().getMainOpcodes());
            }
            if (isRebOnlineAndValid(reb)) {
                try {
                    FirmwareVersion version = getFirmwareVersionForReb(reb);
                    checkAndLoadSequencer(seq, reb, modelAndName, version);
                } catch (REBException | RaftException x) {
                    // TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error stopping or loading sequencer " + modelAndName.getName() + " into " + reb.getFullName(), x);
                }
            }
            // This data will go to the status aggregator in the image-handlers
            data = new SequencerInfo(modelAndName.getName(), String.valueOf(modelAndName.getModel().computeCheckSum()));
            this.sequencerKeyValueData.addData(reb.getFullName(), data);
        }

        this.models = localModels;
        this.registerMap = localRegisterMap;

        if (subsys != null) {
            for (Map.Entry<Reb.RebType, int[]> registersMapEntry : localRegisterMap.entrySet()) {
                sequencerConfig.getCamera().setRegisterList(registersMapEntry.getKey().getLocationType(), registersMapEntry.getValue());
            }
            subsys.publishSubsystemDataOnStatusBus(sequencerKeyValueData);
        }
    }

    private void checkAndLoadSequencer(SequencerProc seq, Reb reb, ModelAndName modelAndName, FirmwareVersion version) throws REBException, RaftException {
        // First check to see if sequencer is running, and if so attempt to stop it
        if (seq.isRunning()) {
            LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
            seq.sendStop();
            // NOTE: waitDone throws a REBException it if times out
            seq.waitDone(2000); // 2 seconds timeout
        }
        boolean rebSupportsMultiMains = version.supportsMultiMains();
        if (supportsMultiMains == null) {
            supportsMultiMains = rebSupportsMultiMains;
        } else if (supportsMultiMains != rebSupportsMultiMains) {
            LOG.log(Level.SEVERE, "Inconsistent firmware version {0} for REB {1}", new Object[]{version, reb.getFullName()});
        }
        LOG.log(Level.INFO, "Loading sequencer {0} into {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
        seq.loadSequencer(modelAndName.getModel());
        sequencerParameterCache.put(reb.getLocation(), loadSequencerParameters(seq, reb, modelAndName.getModel()));
    }

    private void doInBackground(Runnable run) {
        executor.submit(run);
    }

    private void setState(Enum... states) {
        setState(CCSTimeStamp.currentTime(), states);
    }

    private void setState(CCSTimeStamp ts, Enum... states) {
        if (subsys != null) {
            subsys.setState(ts, states);
        }
    }

    Map<SequencerType, ModelAndName> getModels() {
        return models;
    }

    Map<RebType, int[]> getRegisterMap() {
        return registerMap;
    }

    KeyValueDataList getSequencerKeyValueData() {
        return sequencerKeyValueData;
    }

    Map<SequencerType, String> getSequencerPaths() {
        Map<SequencerType, String> sequencerPaths = new HashMap<>();
        for (Entry<SequencerType, ModelAndName> e : models.entrySet()) {
            sequencerPaths.put(e.getKey(), subsys.getAgentInfo().getAgentProperty("org.lsst.ccs.agent.hostname") + ":" + BootstrapResourceUtils.getPathOfResourceInUserResourceDirectories(e.getValue().getName()));
        }
        return sequencerPaths;
    }

    /**
     * Load the compiled sequencer into the specified SequencerProc. This is
     * called when a REB comes online after the initial load of sequencers
     *
     * @param sequencer The SequencerProc to load.
     */
    void load(Reb reb, FirmwareVersion version) {
        if (models != null) {
            SequencerProc seq = REBs.get(reb);
            ModelAndName modelAndName = getModelAndNameForREB(models, reb);
            try {
                checkAndLoadSequencer(seq, reb, modelAndName, version);
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Unable to load sequencer into " + reb.getFullName(), ex);
            }
        }
    }

    private boolean isRebOnlineAndValid(Reb reb) {
        if (subsys == null) {
            return true;
        } else {
            return subsys.isRebOnlineAndValid(reb);
        }
    }

    private FirmwareVersion getFirmwareVersionForReb(Reb reb) {
        if (subsys == null) {
            return new FirmwareVersion(0x31385004);
        } else {
            return subsys.getFirmwareVersionForReb(reb);
        }
    }

    private static ModelAndName getModelAndNameForREB(Map<SequencerType, ModelAndName> models, Reb reb) {
        String rebTypeString = reb.getRebType() + "-" + reb.getCCDType().getName().toUpperCase();
        switch (rebTypeString) {
            case "SCIENCE-E2V":
                ModelAndName result = models.get(SequencerType.SCIENCE_E2V);
                if (result == null) {
                    result = models.get(SequencerType.E2V);
                }
                return result;
            case "SCIENCE-ITL":
            case "WAVEFRONT-ITL": // AuxTel only
                result = models.get(SequencerType.SCIENCE_ITL);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "WAVEFRONT-WF":
                result = models.get(SequencerType.WAVEFRONT);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "GUIDER-ITL":
                result = models.get(SequencerType.GUIDER);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            default:
                throw new IllegalArgumentException("Unknown sequencer type: " + rebTypeString);
        }
    }

    void loadSequencerParameters() {
        REBs.forEach((reb, seq) -> {
            ModelAndName modelAndNameForREB = getModelAndNameForREB(models, reb);
            try {
                sequencerParameterCache.put(reb.getLocation(), loadSequencerParameters(seq, reb, modelAndNameForREB.getModel()));
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Failed to set sequencer parameter for " + reb.getFullName(), ex);
            }
        });
        sequencerConfig.commitBulkChange();
    }

    Map<String, Integer> loadSequencerParameters(SequencerProc seq, Reb reb, FPGA2Model model) throws REBException, RaftException {
        if (isRebOnlineAndValid(reb)) {
            Map<String, Integer> result = sequencerConfig.loadParameters(model, seq);
            if (sequencerConfig.getTranparentMode() != null && subsys != null) {
                subsys.setTransparentMode(reb, sequencerConfig.getTranparentMode());
            }
            return result;
        } else {
            return Collections.EMPTY_MAP;
        }
    }

    LocationSet getConfiguredLocations() {
        return configuredLocations;
    }

    LocationSet getEffectiveLocations() {
        return locations == null || locations.isEmpty() ? configuredLocations : locations;
    }
    
    int getNRebs() {
        return REBs.size();
    }

    private static class ModelAndName {

        private final FPGA2Model model;
        private final String name;

        public ModelAndName(FPGA2Model model, String name) {
            this.model = model;
            this.name = name;
        }

        public FPGA2Model getModel() {
            return model;
        }

        public String getName() {
            return name;
        }
    }
}
