package org.lsst.ccs.subsystem.focalplane;

import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.SupportedOption;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Option;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.data.RaftException;

/**
 * Commands that are used to pause/resume/trigger monitoring.
 *
 * @author The LSST CCS Team
 */
public class RebPowerOnOffCommands {

    private static final Logger LOG = Logger.getLogger(RebPowerOnOffCommands.class.getName());
    
    private final SupportedOption noShortsTestOption = SupportedOption.getSupportedOption("noShortsTest");
    private final SupportedOption dryRunOption = SupportedOption.getSupportedOption("dryRun");
    
    private final FocalPlaneSubsystem focalPlane;

    RebPowerOnOffCommands(FocalPlaneSubsystem focalPlane) {
        this.focalPlane = focalPlane;
    }

    @Option(name = "noShortsTest", description = "If set to true it will skip the shorts test")
    @Option(name = "dryRun", description = "If set to true the command will not execute any commands on the Rebs")
    @Command(description = "Power on a specified set of Rebs", type = Command.CommandType.ACTION, level = 1)
    public void powerOnCCDs(Options options, @Argument String regEx, 
            @Argument(defaultValue = "30", description = "number of seconds to wait between rebs") int delay
            ) throws RaftException {
        Pattern rebPattern = Pattern.compile(regEx);
                
        boolean noShortsTest = options.hasOption(noShortsTestOption);
        boolean dryRun = options.hasOption(dryRunOption);

        //Do we need to check the value of "delay"?
        
        LOG.log(Level.INFO, "Execuring powerOnCCDs for Rebs matching regular expression \"{0}\" with delay of {1} seconds between Rebs. Skip Shorts Test = {2}. Dry Run = {3}", new Object[]{regEx, delay, noShortsTest, dryRun});
        
        for ( Entry<String,REBDevice> e : focalPlane.getRebDevices().entrySet() ) {
            if ( rebPattern.matcher(e.getKey()).matches() ) {

                REBDevice device = e.getValue();                
                if ( device.isBackBiasDisabled() ) {
                    LOG.log(Level.INFO, "Skipping Reb: {0} as it's already on.", e.getKey());                    
                    continue;
                }
                
                LOG.log(Level.INFO, "Starting Power on sequence for Reb: {0}", e.getKey());
                                
                //Perform the shorts test only if requested.
                if ( ! noShortsTest ) {
                    LOG.log(Level.INFO, "Performing shorts test on Reb: {0}", e.getKey());                        
                    if ( !dryRun ) {
                        //test for shorts
                        device.testCCDShorts();

                        //short delay in between 100ms
                        try{Thread.sleep(100L);} catch (InterruptedException ex){}
                    }
                }                

                //Power CCDs on
                LOG.log(Level.INFO, "Invoking powerCCDsOn on Reb: {0}", e.getKey());                        
                if ( !dryRun ) {
                    device.powerCCDsOn();
                }
                
                //Wait the specified amount of time
                try{Thread.sleep(delay*1000L);} catch (InterruptedException ex){}
                
            }
        }
        
    }
    
    @Option(name = "noShortsTest", description = "If set to true it will skip the shorts test")
    @Option(name = "dryRun", description = "If set to true the command will not execute any commands on the Rebs")
    @Command(description = "Power off a specified set of Rebs", type = Command.CommandType.ACTION, level = 1)
    public void powerOffCCDs(Options options, @Argument String regEx, 
            @Argument(defaultValue = "30", description = "number of seconds to wait between rebs") int delay
            ) throws RaftException {
        Pattern rebPattern = Pattern.compile(regEx);
        
        boolean noShortsTest = options.hasOption(noShortsTestOption);
        boolean dryRun = options.hasOption(dryRunOption);

        //Do we need to check the value of "delay"?
        
        LOG.log(Level.INFO, "Execuring powerOffCCDs for Rebs matching regular expression \"{0}\" with delay of {1} seconds between Rebs. Skip Shorts Test = {2}. Dry Run = {3}", new Object[]{regEx, delay, noShortsTest, dryRun});
        
        for ( Entry<String,REBDevice> e : focalPlane.getRebDevices().entrySet() ) {
            if ( rebPattern.matcher(e.getKey()).matches() ) {

                REBDevice device = e.getValue();                
                if ( !device.isBackBiasDisabled() ) {
                    LOG.log(Level.INFO, "Skipping Reb: {0} as it's already of.", e.getKey());                    
                    continue;
                }
                
                LOG.log(Level.INFO, "Starting Power off sequence for Reb: {0}", e.getKey());
                                
                //Perform the shorts test only if requested.
                if ( ! noShortsTest ) {
                    LOG.log(Level.INFO, "Performing shorts test on Reb: {0}", e.getKey());                        
                    if ( !dryRun ) {
                        //test for shorts
                        device.testCCDShorts();

                        //short delay in between 100ms
                        try{Thread.sleep(100L);} catch (InterruptedException ex){}
                    }
                }                

                //Power CCDs on
                LOG.log(Level.INFO, "Invoking powerCCDsOff on Reb: {0}", e.getKey());                        
                if ( !dryRun ) {
                    device.powerCCDsOff();
                }
                
                //Wait the specified amount of time
                try{Thread.sleep(delay*1000L);} catch (InterruptedException ex){}
                
            }
        }
        
    }

}
