package org.lsst.ccs.subsystem.focalplane;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorUpdateTask;

/**
 * The Monitoring configurations for tasks that are to be paused/resumed during
 * readout and triggered right after readout.
 * 
 * @author The LSST CCS Team
 */
public class MonitoringConfig implements ConfigurationBulkChangeHandler {

    private static final Logger LOG = Logger.getLogger(MonitoringConfig.class.getName());

    @ConfigurationParameter(category = "Monitoring", description = "Regular expression to select tasks to pause before readout. An empty String means no tasks will be paused.", units = "unitless")
    public volatile String readoutPauseTasks = "";
    
    @ConfigurationParameter(category = "Monitoring", description = "Regular expression to select tasks to trigger immediately after readout. An empty String means no tasks will be triggered.", units = "unitless")
    public volatile String postReadoutTriggerTasks = "";

    //List of tasks to pause during readout
    private final List<MonitorUpdateTask> tasksToPause = new CopyOnWriteArrayList<>();
    //List of tasks to trigger after readout
    private final List<MonitorUpdateTask> tasksToTrigger = new CopyOnWriteArrayList<>();

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Monitor monitor;
    
    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        //Validate that the provided configuration parameters are valid regular expressions
        //validation happens if the value has changed and the proposed value is not empty
        String proposedReadoutPauseTasks = (String) parametersView.get("readoutPauseTasks");
        if ( !readoutPauseTasks.equals(proposedReadoutPauseTasks) && !proposedReadoutPauseTasks.isEmpty() ) {
            try {
                Pattern.compile(proposedReadoutPauseTasks);
            } catch (PatternSyntaxException e) {
                throw new RuntimeException("The provided \"readoutPauseTasks\" configuration parameter is an invalid regular expression: " + readoutPauseTasks, e);
            }
        }        
        String proposedPostReadoutTriggerTasks = (String) parametersView.get("postReadoutTriggerTasks");
        if ( !postReadoutTriggerTasks.equals(proposedPostReadoutTriggerTasks) && !proposedPostReadoutTriggerTasks.isEmpty() ) {
            try {
                Pattern.compile(proposedPostReadoutTriggerTasks);
            } catch (PatternSyntaxException e) {
                throw new RuntimeException("The provided \"postReadoutTriggerTasks\" configuration parameter is an invalid regular expression: " + postReadoutTriggerTasks, e);
            }
        }
    }
    
    @Override
    public void setParameterBulk(Map<String, Object> parametersView) {
        if ( parametersView.containsKey("readoutPauseTasks") ) {
            String proposedReadoutPauseTasks = (String) parametersView.get("readoutPauseTasks");
            if (!readoutPauseTasks.equals(proposedReadoutPauseTasks)) {
                this.readoutPauseTasks = proposedReadoutPauseTasks;
                tasksToPause.clear();
                if (!readoutPauseTasks.isEmpty()) {
                    Pattern p = Pattern.compile(readoutPauseTasks);
                    tasksToPause.addAll(filterMonitorTasksByName(p));
                }
                LOG.log(Level.INFO, () -> "Configuration parameter \"readoutPauseTasks\" updated to " + readoutPauseTasks + "\n" + printTasksToPauseDuringReadout());
            }
        }

        if ( parametersView.containsKey("postReadoutTriggerTasks") ) {
            String proposedPostReadoutTriggerTasks = (String) parametersView.get("postReadoutTriggerTasks");
            if (!postReadoutTriggerTasks.equals(proposedPostReadoutTriggerTasks)) {
                this.postReadoutTriggerTasks = proposedPostReadoutTriggerTasks;
                tasksToTrigger.clear();
                if (!postReadoutTriggerTasks.isEmpty()) {
                    Pattern p = Pattern.compile(postReadoutTriggerTasks);
                    tasksToTrigger.addAll(filterMonitorTasksByName(p));
                }
                LOG.log(Level.INFO, () -> "Configuration parameter \"postReadoutTriggerTasks\" updated to " + postReadoutTriggerTasks + "\n" + printTasksToTriggerAfterReadout());
            }
        }
    }
    
    private List<MonitorUpdateTask> filterMonitorTasksByName(Pattern p) {
        List<MonitorUpdateTask> result = new ArrayList<>();
        monitor.getMonitorTasksList().stream().filter(t -> (p.matcher(t.getName()).matches())).forEachOrdered(t -> {
            result.add(t);
        });
        return result;
    }
    
    String printTasksToPauseDuringReadout() {
        if ( tasksToPause.isEmpty() ) {
            return "No tasks will be paused during readout.";
        }
        StringBuilder sb = new StringBuilder("The following tasks will be paused during readout: \n");
        int count = 0;
        for (MonitorUpdateTask t : tasksToPause) {
            sb.append("-").append(++count).append("- ").append(t.getName()).append("\n");
        }
        return sb.toString();        
    }
    
    String printTasksToTriggerAfterReadout() {
        if ( tasksToTrigger.isEmpty() ) {
            return "No tasks will be triggered right after readout";
        }
        StringBuilder sb = new StringBuilder("The following tasks will be triggered right after readout: \n");
        int count = 0;
        for (MonitorUpdateTask t : tasksToTrigger) {
            sb.append("-").append(++count).append("- ").append(t.getName()).append("\n");
        }
        return sb.toString();        
    }
    
    void pauseMonitoringTasks() {
        List<Future<Void>> listOfFutures = new ArrayList<>();
        tasksToPause.forEach(t -> {
            listOfFutures.add(t.pausePeriodicUpdate());
        });
        try {
            for (Future<Void> f : listOfFutures) {
                f.get();
            }
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException("Failed to pause monitoring tasks.",e);
        }
    }

    void resumeMonitoringTasks() {
        tasksToPause.forEach(t -> {
            t.resumePeriodicUpdate();
        });        
    }
    
    void triggerMonitoringTasks() {
        tasksToTrigger.forEach(t -> {
            t.scheduleUpdateAndPublishNow();
        });                
    }
    
    List<MonitorUpdateTask> getTasksToPauseDuringReadout() {
        return tasksToPause;
    }
    
    List<MonitorUpdateTask> getTasksToTriggerAfterReadout() {
        return tasksToTrigger;
    }
}
