package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.time.Instant;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Opens a connect to an image database, and updates the database at the completion
 * of each image.
 * @author tonyj
 */
public class ImageDatabase implements AutoCloseable {
    
    private final static int MAX_RETRIES = 1;
    
    private final String dbURL;
    private Connection conn;
    private PreparedStatement stmt1;
    private static final Logger LOG = Logger.getLogger(ImageDatabase.class.getName());
    private final String testStand;
    
    class ImageDAO {
        
        private ImageName imageName;
        private String imageType;
        private String testType;
        private String runNumber;
        private Integer tSeqNum;
        private String fileLocation;
        private Instant obsDate;
        private LocationSet locations;
        private Float exposureTime;
        private Float darkTime;
        private Long imageTag;
        
        void addMetaData(Map<String, Serializable> metaData) {
            metaData.forEach((name, value) -> {
                switch (name) {
                    case "ExposureTime":
                        exposureTime = toFloat(value);
                        break;
                    case "TestSeqNum":
                        tSeqNum = toInt(value);
                        break;
                    case "TestType":
                        testType = toString(value);
                        break;
                    case "ImageType":
                        imageType = toString(value);
                        break;
                    case "RunNumber":
                        runNumber = toString(value);
                        break;
                }
            });
        }
        
        private Float toFloat(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Float) {
                return (Float) value;
            } else if (value instanceof Number) {
                return ((Number) value).floatValue();
            } else {
                return Float.parseFloat(value.toString());
            }
        }
        
        private Integer toInt(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Integer) {
                return (Integer) value;
            } else if (value instanceof Number) {
                return ((Number) value).intValue();
            } else {
                return Integer.parseInt(value.toString());
            }
        }
        
        private String toString(Object value) {
            if (value == null) {
                return null;
            } else {
                return value.toString();
            }
        }
        
        private Date toDate(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Date) {
                return ((Date) value);
            } else if (value instanceof Instant) {
                return java.util.Date.from((Instant) value);
            } else {
                throw new RuntimeException("Cannot convert " + value + " to date");
            }
        }
        
        private void setCurrentImage(ImageName currentImage) {
            this.imageName = currentImage;
        }
        
        void add(FileList fileList) {
            this.fileLocation = fileList.getCommonParentDirectory().toString();
        }
        
        void commit() {
            try {
                insertImage(this);
            } catch (SQLException x) {
                LOG.log(Level.WARNING, x, () -> String.format("failed to write image %s to database", imageName));
            }
        }
        
        @Override
        public String toString() {
            return "ImageDAO{" + "imageName=" + imageName + ", imageType=" + imageType + ", testType=" + testType + ", runNumber=" + runNumber + ", tSeqNum=" + tSeqNum + ", testStand=" + testStand + ", fileLocation=" + fileLocation + ", obsDate=" + obsDate + ", locations=" + locations + ", exposureTime=" + exposureTime + ", darkTime=" + darkTime + ", imageTag=" + (imageTag == null ? imageTag : Long.toHexString(imageTag)) + '}';
        }
        
        void setAnnotation(String annotation) {
            // Currently not used.
        }
        
        void setLocations(LocationSet locations) {
            this.locations = locations;
        }
        
        void setDarkTime(double darkTime) {
            this.darkTime = (float) darkTime;
        }
        
        void setObsDate(Instant taiInstant) {
            this.obsDate = taiInstant;
        }
        
        void setDaqTag(long id) {
            this.imageTag = id;
        }
    }
    
    ImageDatabase(String dbURL, String testStand) {
        this.dbURL = dbURL;
        this.testStand = testStand;
    }
    
    ImageDAO start(ImageName currentImage) {
        ImageDAO dao = new ImageDAO();
        dao.setCurrentImage(currentImage);
        return dao;
    }
    
    static int raftsMaskFromLocations(LocationSet locations) {
        int raftMask = 0;
        if (locations != null) { // Should throw an exception instead?
            for (Location location : locations) {
                raftMask |= 1 << (location.index() / 4);
            }
        }
        return raftMask;
    }
    
    private void openConnection() throws SQLException {
        this.conn = DriverManager.getConnection(dbURL);
        conn.setAutoCommit(false);
        conn.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);
        
        try (PreparedStatement stmt = conn.prepareStatement(
                "create table if not exists ccs_image ("
                + "telCode varchar(2) not null, "
                + "seqnum integer not null, "
                + "dayobs varchar(8) not null, "
                + "controller varchar(2) not null, "
                + "darkTime float, "
                + "exposureTime float, "
                + "fileLocation varchar(255), "
                + "imageTag bigint, "
                + "imgType varchar(20), "
                + "obsDate timestamp, "
                + "raftMask integer, "
                + "runNumber varchar(20), "
                + "testType varchar(20), "
                + "tseqnum integer, "
                + "tstand varchar(20), "
                + "primary key (telCode, seqnum, dayobs, controller), "
                + "index obsDateIndex (obsDate) )")) {
            stmt.execute();
        }
        
        stmt1 = conn.prepareStatement(
                "insert into ccs_image (telcode, seqnum, dayobs, controller, darkTime, exposureTime, fileLocation,"
                + "imageTag, imgType, obsDate, raftMask, runNumber, testType, tseqnum, tstand) "
                + "values (?, ?, ?, ?, ?,"
                + "        ?, ?, ?, ?, ?, "
                + "        ?, ?, ?, ?, ?) ");
    }
    
    void insertImage(ImageDAO dao) throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }
            try {
                stmt1.setString(1, dao.imageName.getSource().getCode());
                stmt1.setInt(2, dao.imageName.getNumber());
                stmt1.setString(3, dao.imageName.getDateString());
                stmt1.setString(4, dao.imageName.getController().getCode());
                stmt1.setObject(5, dao.darkTime);
                stmt1.setObject(6, dao.exposureTime);
                stmt1.setString(7, dao.fileLocation);
                stmt1.setObject(8, dao.imageTag);
                stmt1.setString(9, dao.imageType);
                stmt1.setObject(10, dao.obsDate != null ? Timestamp.from(dao.obsDate) : null);
                stmt1.setInt(11, raftsMaskFromLocations(dao.locations));
                stmt1.setString(12, dao.runNumber);
                stmt1.setString(13, dao.testType);
                stmt1.setObject(14, dao.tSeqNum);
                stmt1.setString(15, testStand);
                stmt1.executeUpdate();
                conn.commit();
                LOG.log(Level.INFO, "Image database updated: {0}", dao.toString());
                break;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }
    
    @Override
    public void close() throws SQLException {
        if (conn != null) {
            conn.close();
            this.conn = null;
        }
    }
    
    // Only used for testing
    Connection getConnection() {
        return conn;
    }
}
