package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.ims.StoreSimulation;
import org.lsst.ccs.daq.ims.StoreSimulation.RegisterAccess;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.GlobalClient;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.drivers.reb.sim.ClientFactorySimulation;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentExecutionService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RebPowerAction;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.focalplane.alerts.FocalPlaneAlertType;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.focalplane.states.GuidingState;
import org.lsst.ccs.subsystem.focalplane.states.SequencerState;
import org.lsst.ccs.subsystem.power.states.RebHvBiasState;
import org.lsst.ccs.subsystem.rafts.AspicControl;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

public class FocalPlaneSubsystem extends Subsystem implements HasLifecycle, HasDataProviderInfos, EventSender, AlertListener {

    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystem.class.getName());

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevices = new LinkedHashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    DataProviderDictionaryService dictionaryService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private SequencerConfig sequencerConfig;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AlertService agentAlertService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ConfigurationService configService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    DataProviderDictionaryService dataProviderDictionaryService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AgentCommandDictionaryService agentCommandDictionaryService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageCoordinatorService imageCoordinatorService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentExecutionService executionService;

    //TO-DO: temporary. Needed by temporary monitoring commands.
    //Remove when commands are removed.
    @LookupField(strategy = LookupField.Strategy.TREE)
    private MonitoringConfig monitoringConfig;

    private Sequencers sequencers;
    private final FocalPlane focalPlaneGeometry;
    private ImageMessageHandling imageMessageHandling;

    private ComponentLookup lookupService;
    private final boolean isSimulationMode;

    //A StatusMessageListener to detect state changes in the HVBias as
    //published from the reb power subsystem
    private StatusMessageListener rebPowerHvBiasStatusMessageListener;

    public FocalPlaneSubsystem(FocalPlane geometry) {
        super("fp", AgentInfo.AgentType.WORKER);
        this.focalPlaneGeometry = geometry;

        Properties props = BootstrapResourceUtils.getBootstrapSystemProperties();
        isSimulationMode = "simulation".equalsIgnoreCase(props.getProperty("org.lsst.ccs.run.mode", "normal"));
        if (isSimulationMode) {
            LOG.log(Level.INFO, "Running in simulation mode");
        }
    }

    @Override
    public void build() {
        agentPropertiesService.setAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, HasFocalPlaneData.generatePropertyValue(FocalPlaneDataGroup.class));
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.FOCAL_PLANE.name());
        agentPropertiesService.setAgentProperty(RebPowerAction.class.getCanonicalName(), "uses");

        // FIXME: What does this do?
        lookupService = this.getComponentLookup();
        ComponentNode thisComponent = lookupService.getComponentNodeForObject(this);

        // This is just avoiding modifying groovy?
        imageMessageHandling = new ImageMessageHandling();
        ComponentNode imageMessageHandlingNode = new ComponentNode("imageMessageHandling", imageMessageHandling);
        lookupService.addComponentNodeToLookup(thisComponent, imageMessageHandlingNode);

        for (Raft r : focalPlaneGeometry.getChildrenList()) {

            ComponentNode raftNode = new ComponentNode(r.getName(), new HardwareIdConfiguration());
            lookupService.addComponentNodeToLookup(thisComponent, raftNode);
            for (Reb reb : r.getRebs()) {
                String rebName = r.getName() + "/" + reb.getName();

                ComponentNode rebHardwareNode = new ComponentNode(reb.getName() + "_hardware", new HardwareIdConfiguration());
                lookupService.addComponentNodeToLookup(raftNode, rebHardwareNode);

                ComponentNode rebComponent = lookupService.getNodeByPath(rebName);
                for (CCD ccd : reb.getCCDs()) {
                    String sensorName = ccd.getName();
                    ComponentNode sensorNode = new ComponentNode(sensorName, new CCDHardwareIdConfiguration());
                    lookupService.addComponentNodeToLookup(rebComponent, sensorNode);
                }
            }
        }
        //Register the FocalPlaneState
        agentStateService.registerState(FocalPlaneState.class, "The state of the Focal Plane", this);
        agentStateService.registerState(SequencerState.class, "The state of the Sequencer", this);
        agentStateService.registerState(GuidingState.class, "The state of the Guider", this);
    }

    @Override
    public void init() {
        // This used to be done by global proc, now we need to do it here.
        ClientFactory clientFactory = isSimulationMode ? new ClientFactorySimulation(true) : new ClientFactory();
        Set<Location> guiderLocations = sequencerConfig.getGuiderLocations().stream().map(sensorLocation -> sensorLocation.getRebLocation()).collect(Collectors.toSet());
        rebDevices.forEach((rebName, rebDevice) -> {
            rebDevice.setClientFactory(clientFactory);
            // Set the partitions for the REBDevices, before they are initialized
            Reb reb = focalPlaneGeometry.getReb(rebName);
            Location rebLocation = reb.getLocation();
            if (guiderLocations.contains(rebLocation)) {
                rebDevice.setIfcName(sequencerConfig.getGuiderPartition());
            } else if (sequencerConfig.getScienceLocations().contains(rebLocation)) {
                rebDevice.setIfcName(sequencerConfig.getPartition());
            } else {
                LOG.log(Level.WARNING, "REBDevice at {0} is neither guider or science", rebLocation);
            }
        });

        if (isSimulationMode) {
            GlobalClient gbl = new GlobalClient();
            gbl.setClientFactory(clientFactory);
            try {
                gbl.open(sequencerConfig.getPartition());
                StoreSimulation.instance().addTriggerListener((int opcode, ImageMetaData meta, Map<Location.LocationType, int[]> registerLists) -> {
                    try {
                        if (meta == null) {
                            gbl.startSequencer(opcode);
                        } else {
                            for (Map.Entry<Location.LocationType, int[]> entry : registerLists.entrySet()) {
                                gbl.setRegisterlist(entry.getKey().getCCDCount(), entry.getValue());
                            }
                            int[] ids = new LocationSet(meta.getLocations()).asIntArray();
                            gbl.acquireImage(meta.getName(), meta.getCreationFolderName(), opcode, meta.getAnnotation(), ids);
                        }
                    } catch (REBException x) {
                        LOG.log(Level.SEVERE, "Error simulating trigger", x);
                    }
                });
                StoreSimulation.instance().setRegisterAccess(new RegisterAccess(){
                    @Override
                    public int readRegister(Location l, int address) {
                        try {
                            REBDevice rebDevice = getRebDeviceAtLocation(l);
                            if (rebDevice != null) {
                                return rebDevice.getRegister(address, 1).getValues()[0];
                            } else {
                                return 0;
                            }
                        } catch (RaftException ex) {
                            return 0;
                        }
                    }

                    @Override
                    public void writeRegister(Location l, int address, int value) {
                        try {
                            getRebDeviceAtLocation(l).setRegister(address, new int[]{value});
                        } catch (RaftException ex) {
                            throw new RuntimeException("Unable to write register", ex);
                        }
                    }

                    private REBDevice getRebDeviceAtLocation(Location l) {
                        return rebDevices.get(l.toString());
                    }
                });
            } catch (REBException x) {
                throw new RuntimeException("Failed to initialize simulation", x);
            }
        }
    }

    @Override
    public void postInit() {

        sequencers = new Sequencers(this, sequencerConfig, executionService);
        agentStateService.updateAgentState(FocalPlaneState.NEEDS_CLEAR, SequencerState.IDLE, GuidingState.UNDEFINED);
        dataProviderDictionaryService.registerClass(ImageMetaDataEvent.class, ImageMetaDataEvent.EVENT_KEY);

        // Loop over all descendant REBDevices. Those which are in scienceLocations are added to sequencers
        LocationSet scienceLocations = sequencerConfig.getScienceLocations();
        rebDevices.forEach((rebName, device) -> {
            Reb reb = focalPlaneGeometry.getReb(rebName);
            device.setRebGeometry(reb);
            device.setIsRealDAQ(sequencerConfig.isRealDAQ());
            if (scienceLocations.contains(reb.getLocation())) {
                LOG.log(Level.INFO, "Adding science REB at {0}", reb.getLocation());
                sequencers.add(reb, device.getSequencer());
            }
        });

        agentStateService.addStateChangeListener(new StateChangeListener() {
            @Override
            public void stateChanged(CCSTimeStamp stateTransitionTimestamp, Object device, Enum<?> newState, Enum<?> oldState) {
                if (newState == RebValidationState.VALID) {
                    final REBDevice rebDevice = (REBDevice) device;
                    String path = lookupService.getNameOfComponent(device);
                    try {
                        FirmwareVersion version = new FirmwareVersion(rebDevice.getHwVersion());
                        if (version.hasLookAtMeBug()) {
                            LOG.log(Level.INFO, "Disabled LAM for firmware {0}", version);
                            // FIXME: Temporary workaround for LSSTCCSRAFTS-529
                            try {
                                rebDevice.setRegister(0x17, new int[]{0});
                            } catch (RaftException x) {
                                LOG.log(Level.SEVERE, "Could not disable LAM for " + rebDevice.getName(), x);
                            }
                        }

                        sequencers.load(focalPlaneGeometry.getReb(path), version);

                        Object result = rebDevice.loadAspics(true);
                        LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{rebDevice.getName(), result});

                    } catch (Exception x) {
                        LOG.log(Level.SEVERE, "Exception while loading ASPICS", x);
                    }
                }
            }
        }, RebDeviceState.class, RebValidationState.class);

        agentCommandDictionaryService.addCommandSetToObject(new LSE71Commands(this, sequencerConfig), "");
        agentCommandDictionaryService.addCommandSetToObject(new ScriptingCommands(this), "");
        agentCommandDictionaryService.addCommandSetToObject(new PlaylistCommands(this, sequencerConfig), "");
        agentCommandDictionaryService.addCommandSetToObject(new FocalPlaneCommands(this), "");
        //TO-DO: Temporary monitoring commands for testing.
        agentCommandDictionaryService.addCommandSetToObject(new MonitoringCommands(monitoringConfig), "");
        agentCommandDictionaryService.addCommandSetToObject(new RebPowerOnOffCommands(this), "");

        //This is needed to publish the Alert summary when a Power Subsystem
        //joins the buses.
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(
                new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo ai : agents) {
                    if (ai.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.POWER.name())) {
                        LOG.log(Level.INFO, "Publishing raised alert summary for {0}", new Object[]{ai.getName()});
                        getScheduler().execute(() -> {
                            agentAlertService.publishRaisedAlertSummary();
                        });
                        break;
                    }
                }
            }
        });
        
        //Add AlertListener to respond to Reb Current Too High Alerts
        agentAlertService.addListener(this);
    }

    @Override
    public void finalizeDictionary() {
        //Manipulate the dictionary here:
        for (DataProviderInfo data : dictionaryService.getDataProviderDictionary().getDataProviderInfos()) {
            FocalPlaneDataGroup dataGroup = FocalPlaneDataGroup.findFocalPlaneDataGroup(data);
            if (dataGroup != null) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }

    @Override
    public void start() {

        rebPowerHvBiasStatusMessageListener = new StatusMessageListener() {

            private final AtomicBoolean isFirstStatusStateChangeNotification = new AtomicBoolean(true);

            @Override
            public void onStatusMessage(StatusMessage msg) {

                //We need to process the states as published by RebPower.
                //The first time we receive any status message we process the full
                //state that comes with the message.
                //After the first message we only process StatusStateChangeNotification
                //messages to analyze only the changes.

                StateBundle stateToProcess = null;
                if ( isFirstStatusStateChangeNotification.getAndSet(false) ) {
                    stateToProcess = msg.getState();
                } else if (msg instanceof StatusStateChangeNotification) {
                    StatusStateChangeNotification stateChange = (StatusStateChangeNotification) msg;
                    stateToProcess = stateChange.getNewState().diffState(stateChange.getOldState());
                }

                if ( stateToProcess != null ) {
                    //Get the components that have the RebHvBiasState
                    Map<String, RebHvBiasState> rebHvBiasStateChanges = stateToProcess.getComponentsWithState(RebHvBiasState.class);
                    if (!rebHvBiasStateChanges.isEmpty()) {
                        for (Entry<String, RebHvBiasState> e : rebHvBiasStateChanges.entrySet()) {


                            RebHvBiasState hvBiasState = e.getValue();
                            //If the RebPower HVBias state is UNKNOWN, don't take any action
                            if (hvBiasState == RebHvBiasState.UNKNOWN) {
                                continue;
                            }
                            REBDevice reb = rebDevices.get(e.getKey());
                            if ( reb != null && !reb.isOnline() ) {
                                continue;
                            }
                            if (reb != null) {
                                if (hvBiasState == RebHvBiasState.OFF || hvBiasState == RebHvBiasState.ON) {
                                    try {
                                        reb.disableBackBias(hvBiasState == RebHvBiasState.OFF, "Power subsystem ("+msg.getOriginAgentInfo().getName()+") turned HVBias " + hvBiasState.name() + " for " + reb.getPath());
                                    } catch (RaftException re) {
                                        LOG.log(Level.WARNING, "Exception when disabling reb "+reb.getPath(),re);
                                    }

                                }
                            } else {
                                LOG.log(Level.WARNING, "Could not find reb {0}.", e.getKey());
                            }
                        }
                    }
                }
            }
        };

        //Add the HVBias status message listener; it only listens to agents
        //of category POWER
        getMessagingAccess().addStatusMessageListener( rebPowerHvBiasStatusMessageListener
                , (m)-> {return m.getOriginAgentInfo().getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.POWER.name());});

    }


    @Override
    public void shutdown() {
        //Remove the HVBias status message listener
        getMessagingAccess().removeStatusMessageListener(rebPowerHvBiasStatusMessageListener);
    }


    @Override
    public void postStart() {
        configService.addConfigurationListener(new ConfigListener(this));
        loadSequencers();
    }

    void loadSequencers() {
        try {
            sequencers.load();
        } catch (RaftException | REBException | DAQException x) {
            //TODO: Go into fault state?
            throw new RuntimeException("Error configuring sequencers", x);
        }
    }

    void loadASPICS() {
        for (REBDevice device : rebDevices.values()) {
            if (device.isOnline() && device.isSerialNumValid()) {
                try {
                    Object result = device.loadAspics(true);
                    LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{device.getFullName(), result});
                } catch (Exception ex) {
                    //TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error while loading ASPICS", ex);
                }
            }
        }
    }

    boolean isRebOnlineAndValid(Reb reb) {
        return stateService.isComponentInState(reb.getFullName(), RebValidationState.VALID);
    }

    FirmwareVersion getFirmwareVersionForReb(Reb reb) {
        return new FirmwareVersion(rebDevices.get(reb.getFullName()).getHwVersion());
    }

    public Sequencers getSequencers() {
        return sequencers;
    }

    ImageNameService getImageNameService() {
        return imageNameService;
    }

    Map<String, REBDevice> getRebDevices() {
        return rebDevices;
    }

    void loadSequencerParameters() {
        sequencers.loadSequencerParameters();
    }

    @Override
    public void sendEvent(String key, Serializable event) {
        KeyValueData kvd = new KeyValueData(key, event);
        publishSubsystemDataOnStatusBus(kvd);
        LOG.log(Level.INFO, "Sent: {0}", event);
    }

    void setState(FocalPlaneState newState) {
        agentStateService.updateAgentState(newState);
    }

    void setStateIf(Enum currentState, Enum... newStates) {
        synchronized (agentStateService.getStateLock()) {
            if (agentStateService.getState(currentState.getClass()) == currentState) {
               agentStateService.updateAgentState(newStates);
            }
        }
    }

    boolean isInState(Enum... states) {
        synchronized (agentStateService.getStateLock()) {
            for (Enum state : states) {
                if (!agentStateService.isInState(state)) return false;
            }
            return true;
        }
    }

    void setState(CCSTimeStamp ts, Enum... stateChanges) {
         agentStateService.updateAgentState(ts, stateChanges);
    }

    ImageCoordinatorService getImageCoordinatorService() {
        return imageCoordinatorService;
    }

    /**
     * Method called to add meta-data to images. The meta-data can got both into the image-handlers, and
     * into the image image database.
     * @param headersMap
     */
    void setHeaderKeywords(Map<String, Serializable> headersMap) {
        imageCoordinatorService.addMetaData(headersMap);
    }

    /**
     * Ugly way to put the ASPICS into transparent mode. Required for LSSTCCSRAFTS-567.
     * @param reb
     * @param tranparentMode
     */
    void setTransparentMode(Reb reb, boolean tranparentMode) {
        rebDevices.get(reb.getFullName()).setAllAspic(AspicControl.TM, tranparentMode ? 1 : 0, false);
    }

    public boolean isSimulationMode() {
        return isSimulationMode;
    }


    private final String rebBoardCurrentTooHighAlertId = FocalPlaneAlertType.REB_BOARD_CURRENT_TOO_HIGH.getAlertId()+"/";
    
    /** 
     * Alert response for internally raised alerts;
     * 
     * @param event 
     */
    @Override
    public void onAlert(AlertEvent event) {
        if ( event.getType() == AlertEvent.AlertEventType.ALERT_RAISED ) {
            Alert a = event.getAlert();            
            String alertId = a.getAlertId();
            if ( alertId.startsWith(rebBoardCurrentTooHighAlertId) && event.getLevel() == AlertState.ALARM ) {
                String rebPath = alertId.replace(rebBoardCurrentTooHighAlertId, "");
                try {                    
                    rebDevices.get(rebPath).powerCCDsOff();                        
                    LOG.log(Level.WARNING, "The CCDs on REB {0} have been turned off in response to a High Current Alert.", rebPath);                    
                } catch (RaftException e) {
                    LOG.log(Level.SEVERE, "Failed to turn off CCDs on REB "+rebPath+" in response to a High Current Alert", e);                                        
                }
            }            
        }
    }
   
    
    
}
