package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;

/**
 * Commands that are when using daq emulation or simulation to control DAQ playlists
 *
 * @author tonyj
 */
public class PlaylistCommands {

    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig config;
    private static final String PLIST_SUFFIX = ".plist";

    private static final Logger LOG = Logger.getLogger(PlaylistCommands.class.getName());

    PlaylistCommands(FocalPlaneSubsystem subsys, SequencerConfig config) {
        this.subsys = subsys;
        this.config = config;
    }

    /**
     * Start a playlist. Note this method now works by tracking the play list
     * item inside CCS, and spoon feeding them to the DAQ as single image
     * playlists.
     *
     * @param playListFileName The play list name
     * @param repeat Whether the playlist should be started in repeat mode
     * @throws java.io.IOException
     */
    @Command(type = Command.CommandType.ACTION, description = "Start a playlist", level = Command.NORMAL, autoAck = false)

    public void play(String playListFileName, @Argument(defaultValue = "false") boolean repeat) throws IOException {
        Path playListPath = getPlaylistBaseDir(playListFileName);

        subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(!config.isRealDAQ(), "play command only evailable with emulated or simulated DAQ")
                .precondition(Files.exists(playListPath), "play list backing file does not exist")
                .action(() -> {
                    CCSImageList imageList = CCSImageList.open(playListPath);
                    CCSPlayList cpl = config.hasSimulatedDAQ() ? 
                            new CCSSimulatedPlayList(imageList, repeat, subsys,  config) :
                            new CCSEmulatedPlayList(imageList, repeat, config.getStore());
                    cpl.validate();
                    config.setCurrentPlaylist(cpl);
                    LOG.log(Level.INFO, "Playlist set {0}", cpl);
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "Define a playlist", level = Command.NORMAL, autoAck = false)
    public void definePlaylist(String playListFileName, String daqFolder, String... imageNames) {

        subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(!config.isRealDAQ(), "playlists only evailable with emulated or simulated DAQ")
                .action(() -> {
                    Path playListPath = getPlaylistBaseDir(playListFileName);
                    if (Files.deleteIfExists(playListPath));
                    CCSImageList imageList = CCSImageList.create(playListPath, config.hasSimulatedDAQ() ? config.getSimulatedDataFolder() : "DAQ", daqFolder);
                    for (String imageName : imageNames) {
                        imageList.add(imageName);
                    }
                    imageList.save();
                    return null;
                });
    }


    @Command(type = Command.CommandType.QUERY, description = "Show contents of a playlist", level = Command.NORMAL, autoAck = false)
    public String showPlaylist(String playListFileName) throws IOException {
        Path playListPath = getPlaylistBaseDir(playListFileName);

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(!config.isRealDAQ(), "playlists only evailable with emulated or simulated DAQ")
                .precondition(Files.exists(playListPath), "play list backing file does not exist")
                .action(() -> {
                    CCSImageList imageList = CCSImageList.open(playListPath);
                    List<String> images = imageList.getImages();
                    return images.stream().collect(Collectors.joining("\n"));
                });
    }

    @Command(type = Command.CommandType.QUERY, description = "Show status of the current playlist", level = Command.NORMAL, autoAck = false)
    public String showCurrentPlaylist() throws IOException {

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(!config.isRealDAQ(), "playlists only evailable with emulated or simulated DAQ")
                .precondition(config.getCurrentPlaylist() != null, "no playlist currently active")
                .action(() -> {
                    return config.getCurrentPlaylist().toString();
                });
    }

    @Command(type = Command.CommandType.QUERY, description = "Show status of the current playlist", level = Command.NORMAL, autoAck = false)
    public String listPlaylists() throws IOException {

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(!config.isRealDAQ(), "playlists only evailable with emulated or simulated DAQ")
                .action(() -> {
                    List<Path> playlists = Files.list(config.getPlaylistBaseDir())
                            .filter(p -> p.getFileName().toString().endsWith(PLIST_SUFFIX))
                            .collect(Collectors.toList());
                    List<String> results = new ArrayList<>();
                    for (Path path : playlists) {
                        CCSImageList playList = CCSImageList.open(path);
                        results.add(String.format("%s: size=%d", playList.getName(), playList.size()));
                    }
                    Collections.sort(results);
                    return results.stream().collect(Collectors.joining("\n"));
                });
    }

    private Path getPlaylistBaseDir(String playlistFileName) throws IOException {
        if (!playlistFileName.endsWith(PLIST_SUFFIX)) {
            playlistFileName += PLIST_SUFFIX;
        }
        return config.getPlaylistBaseDir().resolve(playlistFileName);
    }

}
