package org.lsst.ccs.subsystem.focalplane;

import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.ReadoutMode;
import org.lsst.ccs.subsystem.imagehandling.data.AdditionalFile;
import org.lsst.ccs.subsystem.imagehandling.data.JsonFile;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.location.SensorLocation;
import org.lsst.ccs.utilities.location.SensorLocationSet;

/**
 * Send the list of expected sensors to image handler, for communication with
 * DM/RapidAnalaysis
 *
 * @author tonyj
 */
public class ExpectedSensorsEventSender {

    private static final Logger LOG = Logger.getLogger(ExpectedSensorsEventSender.class.getName());

    private final EventSender sender;
    private final SensorLocationSet scienceLocations = new SensorLocationSet();
    private final SensorLocationSet guiderLocations = new SensorLocationSet();
    private ImageName obsId;

    ExpectedSensorsEventSender(EventSender sender) {
        this.sender = sender;
    }

    void startGuiding(ImageName imageName, SensorLocationSet guiderLocations) {
        this.obsId = imageName;
        this.guiderLocations.clear();
        this.guiderLocations.addAll(guiderLocations);
    }

    void stopGuiding() {
        if (scienceLocations.isEmpty()) {
            sendEvent();
        }
    }

    void startIntegration(ImageName imageName, LocationSet locations) {
        this.obsId = imageName;
        scienceLocations.clear();
        for (Location location : locations) {
            scienceLocations.addAll(location.getSensors());
        }
    }

    void endIntegration(ReadoutMode mode) {
        if (mode != ReadoutMode.TRUE) {
            scienceLocations.clear();
        }
        if (!scienceLocations.isEmpty() || guiderLocations.isEmpty()) {
            sendEvent();
        }
    }

    private void sendEvent() {
        Map<String, String> jsonData = new TreeMap<>();
        for (SensorLocation l : scienceLocations) {
            jsonData.put(l.toString("_"), "SCIENCE");
        }
        for (SensorLocation l : guiderLocations) {
            jsonData.put(l.toString("_"), "GUIDER");
        }
        JsonFile json = new JsonFile(obsId+"_expectedSensors.json", "expectedSensors", obsId, "expectedSensors", jsonData);
        LOG.log(Level.INFO, "Sending expectedSensors: {0}", json);
        sender.sendEvent(AdditionalFile.EVENT_KEY, json);
        scienceLocations.clear();
        guiderLocations.clear();
    }
}
