package org.lsst.ccs.subsystem.focalplane;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.LockSupport;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.drivers.reb.BaseSet;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * A utility for tracking REB devices, used by Sequencers 
 * @author tonyj
 */
public class RebDevices {

    private static final Logger LOG = Logger.getLogger(RebDevices.class.getName());

    private final SequencerConfig config;
    private final LocationSet locations;
    private final SequencerMonitor sequencerMonitor;
    private final ReentrantLock sequencerLock = new ReentrantLock();
    private final Condition sequencerStopped = sequencerLock.newCondition();
    private final ConcurrentLinkedQueue<Runnable> stoppedQueue = new ConcurrentLinkedQueue<>();
    private volatile boolean isStarted = false;
    private final ExecutorService executor;

    public RebDevices(ExecutorService executor, SequencerConfig config, LocationSet locations) {
        this.config = config;
        this.locations = locations;
        this.executor = executor;
        this.sequencerMonitor = new SequencerMonitor();
    }

    private void start() {
        synchronized (sequencerMonitor) {
            if (!isStarted) {
                executor.execute(sequencerMonitor);
                isStarted = true;
            }
        }
    }

    boolean areSequencersStopped() throws DAQException {
        Map<Location, int[]> statusRegisters = config.getRegisterClient().readRegisters(locations, BaseSet.REG_STATE);
        for (Location l : locations) {
            if ((statusRegisters.get(l)[0] & (1 << BaseSet.RSET_SEQUENCER)) != 0) {
                return false;
            }
        }
        return true;
    }

    void waitSequencersStopped(Duration timeout) throws REBException {
        if (!isStarted) {
            start();
        }
        try {
            boolean stopped = sequencerMonitor.waitSequencersStopped(timeout);
            if (!stopped) {
                throw new REBException("Timedout waiting for sequencers to stop");
            }
        } catch (InterruptedException x) {
            throw new REBException("Interrupt while waiting for sequencers to stop");
        }
    }

    void whenSequencersStopped(Runnable runMe) {
        if (!isStarted) {
            start();
        }
        stoppedQueue.offer(runMe);
    }

    private class SequencerMonitor implements Runnable {

        private Thread runThread;
        private final long nanos = 1_000_000;

        @Override
        public void run() {
            runThread = Thread.currentThread();
            runThread.setName("Sequencer wait thread");
            long loopStart = System.nanoTime();
            for (int i = 0;; i++) {
                try {
                    long start = System.nanoTime();
                    int queueLength = 0;
                    boolean stopped = areSequencersStopped();
                    if (stopped) {
                        for (;;) {
                            Runnable r = stoppedQueue.poll();
                            if (r == null) {
                                break;
                            }
                            r.run();
                        }
                        sequencerLock.lock();
                        try {
                            sequencerStopped.signalAll();
                            queueLength = 0;
                        } finally {
                            sequencerLock.unlock();
                        }
                    } else if (queueLength == 0) {
                        sequencerLock.lock();
                        try {
                            queueLength = sequencerLock.getWaitQueueLength(sequencerStopped);
                        } finally {
                            sequencerLock.unlock();
                        }
                    }
                    long nanosToWait = nanos; // queueLength > 0 ? nanos / 10 : nanos;
                    for (;;) {
                        long timeToWait = nanosToWait - (System.nanoTime() - start);
                        if (timeToWait < 0) {
                            break;
                        }
                        LockSupport.parkNanos(timeToWait);
                    }
                    if (i % 1000 == 0) {
                        final long loopTime = System.nanoTime() - loopStart;
                        final int q = queueLength;
                        final int count = i;
                        LOG.log(Level.INFO, () -> String.format("RebDevices Loop %d: %,dns with queue length %d", count, loopTime, q));
                        loopStart = System.nanoTime();
                    }
                } catch (Throwable x) {
                    LOG.log(Level.SEVERE, "DAQ exception in REBDevices loop", x);
                }
            }
        }

        boolean waitSequencersStopped(Duration timeout) throws InterruptedException {
            sequencerLock.lock();
            try {
                LockSupport.unpark(runThread);
                return sequencerStopped.await(timeout.toMillis(), TimeUnit.MILLISECONDS);
            } finally {
                sequencerLock.unlock();
            }
        }
    }
}
