package org.lsst.ccs.subsystem.focalplane;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.guider.ROICommon;
import org.lsst.ccs.daq.guider.ROISpec;
import org.lsst.ccs.daq.guider.Series;
import org.lsst.ccs.daq.guider.Status;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Guider;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.GuidingState;
import org.lsst.ccs.utilities.location.SensorLocationSet;

/**
 *
 * @author tonyj
 */
public class Guiding {

    private static final Logger LOG = Logger.getLogger(Guiding.class.getName());

    private final SequencerConfig sequencerConfig;
    private final AgentStateService agentStateService;
    private ROISpec roiSpec;
    private final static SensorLocationSet NO_SENSORS = new SensorLocationSet();

    public Guiding(SequencerConfig config, AgentStateService agentStateService) {
        this.sequencerConfig = config;
        this.agentStateService = agentStateService;
        boolean hasGuiderPartition = sequencerConfig.hasGuiderPartition();
        if (hasGuiderPartition) {
            try {
                updateState(sequencerConfig.getGuider().series().getStatus().getOut());
            } catch (DAQException x) {
                LOG.log(Level.SEVERE, "Error initializing guider state", x);
                agentStateService.updateAgentState(GuidingState.UNDEFINED);
            }
        } else {
            agentStateService.updateAgentState(GuidingState.UNDEFINED);
        }
    }

    private void updateState(Status.State state) {
        LOG.log(Level.INFO, "Initial guding state {0}", state);
        agentStateService.updateAgentState(GuidingState.values()[state.ordinal()]);
    }

    void initGuiders(String roiSpec) throws DAQException {
        this.roiSpec = ROISpec.parse(roiSpec);
    }

    /**
     * 
     * @param roiSpec The roispec to validate
     * @return The estimated time between stamps, used to estimate when the guiding needs to be stopped
     * @throws DAQException 
     */
    int validate(String roiSpec) throws DAQException {
        ROISpec spec = ROISpec.parse(roiSpec);
        Guider guider = sequencerConfig.getGuider();
        spec.sanityCheck(guider.getConfiguredLocations());
        ROICommon common = spec.getCommon();
        guider.validate(common, spec.getLocations());
        //  Per Gregg: Sequencer_time_in_ns = 32751760 + 490700*RoiRows + 1030*RoiCols*RoiRows
        int rows = common.getRows();
        int cols = common.getCols();
        int ns = 32751760 + 490700*rows + 1030*cols*rows; 
        return common.getIntegrationTimeMillis() + ns/1_000_000;
    }

    /**
     * Start guiding using the previously set ROIs
     * @param imageName The current imageName
     * @throws DAQException 
     */
    void startGuiding(String imageName) throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        final Series series = guider.series();
        Status.State out = series.getStatus().getOut();
        if (out == Status.State.OFF) {
            guider.wake();
        }
        guider.start(roiSpec.getCommon(), imageName, roiSpec.getLocations());
        guider.resume();
        agentStateService.updateAgentState(GuidingState.RUNNING);
    }

    /**
     * Stop guiding
     * @return The number of stamps accumulated since startGuiding
     * @throws DAQException 
     */
    int stopGuiding() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        guider.pause();
        int stamps = guider.series().getStamps();
        guider.stop();
        agentStateService.updateAgentState(GuidingState.STOPPED);
        return stamps;
    }
    
    boolean isROISet() {
        return roiSpec != null;
    }

    void clearROI() {
        roiSpec = null;
    }

    SensorLocationSet getSensors() {
        return roiSpec == null ? NO_SENSORS : roiSpec.getSensorLocations();
    }

    void sleepGuider() throws DAQException {
        Guider guider = sequencerConfig.getGuider();
        guider.sleep();
        agentStateService.updateAgentState(GuidingState.OFF);
    }
}
