package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * A CCS implementation of a playlist on top of DAQ single image playlists
 * @author tonyj
 */
abstract class CCSPlayList {

    private static final Logger LOG = Logger.getLogger(CCSPlayList.class.getName());

    private final CCSImageList imageList;
    private final boolean repeat;
    private int index;

    /**
     * Create a CCS playlist
     * @param playListFileName
     * @param repeat
     * @param emulator
     * @throws IOException 
     */
    CCSPlayList(CCSImageList imageList, boolean repeat) {

        this.imageList = imageList;
        this.repeat = repeat;
        this.index = 0;
    }
    
    String getName() {
        return imageList.getName();
    }

    boolean hasNextImage() {
        return index < imageList.size();
    }

    /**
     * Get the next image in the play list, without incrementing
     * @return The next image, or null of there is no next image
     */
    String peekNextImage() {
        return (hasNextImage()) ? imageList.get(index) : null;
    }

    void playNextImage(String triggeredImageName, LocationSet effectiveLocations,  int opCode, String annotation) throws IOException, DAQException {
        if (!hasNextImage()) {
            throw new IOException("Playlist exhausted "+this);
        }
        String currentImageName = imageList.get(index++);
        if (index >= imageList.size() && repeat) {
            index = 0;
        }
        play(currentImageName, triggeredImageName, effectiveLocations, opCode, annotation);
    }

    @Override
    public String toString() {
        String next = peekNextImage();
        return "CCSPlayList{" + "name=" + getName() + ", index=" + index + ", size=" + imageList.size() + ", repeat=" + repeat + ", nextImage=" + (next==null ? "None" : next) + '}';
    }

    abstract void play(String emulatedImageName, String triggeredImageName, LocationSet effectiveLocations,  int opCode, String annotation) throws DAQException, IOException;
}
