package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.daq.ims.Camera;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Emulator;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.daq.ims.RegisterClient;
import org.lsst.ccs.daq.ims.Store;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.services.AgentExecutionService;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.utilities.readout.ReadOutParametersNew;

/**
 * The sequencer configuration object
 *
 * @author tonyj
 */
@SuppressWarnings("FieldMayBeFinal")
public class SequencerConfig implements ConfigurationBulkChangeHandler {

    private static final Logger LOG = Logger.getLogger(SequencerConfig.class.getName());

    private static final int UNDEFINED = -1;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentExecutionService executionService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService sce;

    @ConfigurationParameter(category = "Sequencer", maxLength = 3)
    private volatile Map<SequencerType, String> sequencer = new HashMap<>();

    /**
     * This is a read-only configuration parameter with the checksum for each
     * sequencer. It is used by the OCSBridge to fill in the SAL settingsApplied
     * event.
     */
    @ConfigurationParameter(category = "Sequencer", isReadOnly = true, maxLength = 75)
    private volatile Map<SequencerType, Long> sequencerChecksums = new HashMap<>();

    @ConfigurationParameter(category = "Sequencer", description = "Names of meta-data related registers which must be in the sequencer", maxLength = 16)
    private volatile String[] metaDataRegisters = ReadOutParametersNew.DEFAULT_NAMES;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int preCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int underCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int readCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int postCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int readCols2 = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int overCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int preRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int readRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int postRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private volatile int overRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer", description = "True if the sequencer is in scan mode", units = "unitless")
    private volatile boolean scanMode = false;

    @ConfigurationParameter(category = "Sequencer", description = "True if the ASPIC is in transparent mode", units = "unitless")
    private volatile int transparentMode = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer", description = "Name of the clear main", units = "unitless")
    private volatile String clearMain = "Clear";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the idle flush main", units = "unitless")
    private volatile String idleFlushMain = "Idle";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the integrate main", units = "unitless")
    private volatile String integrateMain = "Integrate";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the row shift reverse main", units = "unitless")
    private volatile String rowShiftReverseMain = "RowShiftR";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the row shift forward main", units = "unitless")
    private volatile String rowShiftForwardMain = "RowShiftF";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the read main", units = "unitless")
    private volatile String readMain = "Read";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the pseudo read main", units = "unitless")
    private volatile String pseudoReadMain = "PseudoRead";

    @ConfigurationParameter(category = "Sequencer", description = "Idle time in milliseconds before sequencer will enter IDLE_FLUSH. Can be zero (enter immeidately) or negative (never)",
            units = "unitless")
    private volatile long idleFlushTimeout = -1;

    @ConfigurationParameter(category = "Sequencer", description = "Name of the clear count parameter", units = "unitless")
    private volatile String clearCountParameter = "ClearCount";
    @ConfigurationParameter(category = "Sequencer", description = "Name of the shift count parameter", units = "unitless")
    private volatile String shiftCountParameter = "ShiftCount";

    @ConfigurationParameter(category = "Sequencer", description = "Read all registers in parallel when waiting for sequencers to stop", units = "unitless")
    private volatile boolean useParallelRegisters = false;

    @ConfigurationParameter(category = "Sequencer", description = "Perform a step rather than stop after integrate", units = "unitless")
    private volatile boolean stepAfterIntegrate = false;

    @ConfigurationParameter(category = "DAQ", description = "Folder in 2-day store where images will be created", units = "unitless")
    private volatile String daqFolder = "raw";

    @ConfigurationParameter(category = "DAQ", description = "DAQ partition", units = "unitless")
    private volatile String daqPartition = "camera";

    @ConfigurationParameter(category = "DAQ", description = "True when using an emulated DAQ", units = "unitless")
    private volatile boolean emulatedDAQ = false;

    @ConfigurationParameter(category = "DAQ", description = "Number of bad pixels allowed before alarm raised", units = "unitless")
    private volatile int badPixelAlarmLimit = 20000;

    private volatile List<ImageListener> imageListeners = new CopyOnWriteArrayList<>();
    private volatile Map<SequencerType, Long> newChecksums;
    private volatile Store store;
    private CCSPlayList currentPlaylist;

    public Map<SequencerType, String> getSequencers() {
        return sequencer;
    }

    public String[] getMetaDataRegisters() {
        return metaDataRegisters;
    }

    // For test purposes only
    void setTestSequencers(Map<SequencerType, String> sequencers) {
        this.sequencer = sequencers;
        this.pseudoReadMain = "PsuedoRead";
    }

    public String getDAQFolder() {
        return daqFolder;
    }

    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        Map<SequencerType, String> sequencerMap = (Map<SequencerType, String>) parametersView.get("sequencer");
        if (sequencerMap.isEmpty()) {
            throw new IllegalArgumentException("Invalid empty sequencer map");
        }
        // We compute the checksums during validation, and then "commit" them when the new configuration is accepted.
        newChecksums = new HashMap<>();
        sequencerMap.forEach((sequencerType, SequencerName) -> {
            try {
                List<String> mains = Arrays.asList(
                        parametersView.get("clearMain").toString(),
                        parametersView.get("idleFlushMain").toString(),
                        parametersView.get("integrateMain").toString(),
                        parametersView.get("pseudoReadMain").toString(),
                        parametersView.get("readMain").toString(),
                        parametersView.get("rowShiftForwardMain").toString(),
                        parametersView.get("rowShiftReverseMain").toString());
                List<String> parameters = new ArrayList(Arrays.asList((String[]) parametersView.get("metaDataRegisters")));
                parameters.addAll(Arrays.asList(
                        parametersView.get("clearCountParameter").toString(),
                        parametersView.get("shiftCountParameter").toString()));
                FPGA2Model model = Sequencers.validate(mains, parameters, SequencerName);
                newChecksums.put(sequencerType, model.computeCheckSum());
            } catch (IOException | REBException | RaftException ex) {
                throw new IllegalArgumentException("Invalid sequencer file " + SequencerName, ex);
            }
        });
    }

    @Override
    public void setParameterBulk(Map<String, Object> parametersView) {
        if (parametersView.containsKey("daqPartition")) {
            try {
                // If the partition has been changed we need to close the old partition
                if (store != null) {
                    store.close();
                    // And if there are image listeners we need to transfer them to the new partition
                    if (!imageListeners.isEmpty()) {
                        store = new Store(parametersView.get("daqPartition").toString(), executionService);
                        for (ImageListener l : imageListeners) {
                            store.addImageListener(l);
                        }
                    } else {
                        store = null;
                    }
                }
            } catch (DAQException ex) {
                LOG.log(Level.WARNING, "Unable to close old DAQ store", ex);
            }
        }
        ConfigurationBulkChangeHandler.super.setParameterBulk(parametersView);
        sequencerChecksums = newChecksums;

    }

    Map<String, Integer> loadParameters(FPGA2Model model, SequencerProc seq) throws REBException, RaftException {
        Map<String, Integer> cache = new LinkedHashMap<>();
        setParameter(seq, model, cache, "PreCols", preCols);
        setParameter(seq, model, cache, "UnderCols", underCols);
        setParameter(seq, model, cache, "ReadCols", readCols);
        setParameter(seq, model, cache, "PostCols", postCols);
        setParameter(seq, model, cache, "ReadCols2", readCols2);
        setParameter(seq, model, cache, "OverCols", overCols);
        setParameter(seq, model, cache, "PreRows", preRows);
        setParameter(seq, model, cache, "ReadRows", readRows);
        setParameter(seq, model, cache, "PostRows", postRows);
        setParameter(seq, model, cache, "OverRows", overRows);
        seq.enableScan(scanMode);
        return cache;
    }

    private void setParameter(SequencerProc seq, FPGA2Model model, Map<String, Integer> cache, String name, int value) throws REBException, RaftException {
        PointerInfo pi = model.getPointerMap().get(name);
        if (pi == null) {
            throw new RaftException("Invalid pointer " + name);
        }
        final int actualValue = value != UNDEFINED ? value : pi.getValue();
        seq.setParameter(name, actualValue);
        cache.put(name, actualValue);
    }

    boolean hasEmulatedDAQ() {
        return emulatedDAQ;
    }

    boolean isScanMode() {
        return scanMode;
    }

    Store getStore() throws DAQException {
        if (store == null) {
            store = new Store(daqPartition, executionService);
        }
        return store;
    }

    Camera getCamera() throws DAQException {
        return getStore().getCamera();
    }

    Emulator getEmulator() throws DAQException {
        return getStore().getEmulator();
    }

    RegisterClient getRegisterClient() throws DAQException {
        return getStore().getRegisterClient();
    }

    String getPartition() {
        return daqPartition;
    }

    boolean isTransparentOrScanMode() {
        return scanMode || transparentMode == 1;
    }
    
    /**
     * Check if we are in transparent mode, note this method returns <code>null</code>
     * if transparent mode is UNDEFINED(-1) in the configuration.
     * @return true, false or null
     */
    Boolean getTranparentMode() {
        return transparentMode == UNDEFINED ? null : transparentMode == 1;
    }

    public String getClearMain() {
        return clearMain;
    }

    public String getIdleFlushMain() {
        return idleFlushMain;
    }

    public String getIntegrateMain() {
        return stepAfterIntegrate ? "IntegrateRead" : integrateMain;
    }

    public String getRowShiftReverseMain() {
        return rowShiftReverseMain;
    }

    public String getRowShiftForwardMain() {
        return rowShiftForwardMain;
    }

    public String getPseudoReadMain() {
        return pseudoReadMain;
    }

    public String getClearCountParameter() {
        return clearCountParameter;
    }

    public List<String> getRequiredMains() {
        return Arrays.asList(clearMain, idleFlushMain, integrateMain, pseudoReadMain, readMain, rowShiftForwardMain, rowShiftReverseMain);
    }

    public List<String> getRequiredParameters() {
        List<String> result = new ArrayList(Arrays.asList(metaDataRegisters));
        result.addAll(Arrays.asList(clearCountParameter, shiftCountParameter));
        return result;
    }

    public String getShiftCountParameter() {
        return shiftCountParameter;
    }

    public String getReadMain() {
        return readMain;
    }

    public long getIdleFlushTimeout() {
        return idleFlushTimeout;
    }

    public boolean useParallelRegisters() {
        return useParallelRegisters;
    }

    public boolean useStepAfterIntegrate() {
        return stepAfterIntegrate;
    }

    public String getIntegrateAfterPointer() {
        return "AfterIntegrate";
    }

    public String getReadSubroutine() {
        return "ReadFrame";
    }

    String getPseudoSubroutine() {
        return "PseudoFrame";
    }

    String getNoOpSubroutine() {
        return "NoOp";
    }

    void commitBulkChange() {
        sce.commitBulkChange();
    }

    int getBadPixelAlarmLimit() {
        return badPixelAlarmLimit;
    }

    Path getPlaylistBaseDir() throws IOException {
        Path patternDir = Paths.get(System.getProperty("user.home"), "playlists");
        Files.createDirectories(patternDir);
        return patternDir;
    }
    // We keep track of listeners here, so if the DAQ partition is changed, we
    // can continue to listen.
    void addDAQImageListener(ImageListener imageListener) throws DAQException {
        getStore().addImageListener(imageListener);
        imageListeners.add(imageListener);
    }

    void removeDAQImageListener(ImageListener imageListener) throws DAQException {
        getStore().removeImageListener(imageListener);
        imageListeners.remove(imageListener);
    }

    CCSPlayList getCurrentPlaylist() {
        return currentPlaylist;
    }

    void setCurrentPlaylist(CCSPlayList cpl) {
        this.currentPlaylist = cpl;
    }
}
