package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;

/**
 * Commands that are useful when running the focal plane subsystem with
 * scripting
 *
 * @author tonyj
 */
public class ScriptingCommands {

    private final FocalPlaneSubsystem subsys;

    private static final Logger LOG = Logger.getLogger(ScriptingCommands.class.getName());

    ScriptingCommands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    /**
     * Wait until all FITS files have been written.
     *
     * @param timeout The maximum duration to wait
     * @return The list of files which were written
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for FITS files to be saved", level = Command.NORMAL, autoAck = false)
    public FileList waitForFitsFiles(@Argument(defaultValue = "PT30S") Duration timeout) {
        return subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> {
                    return subsys.getImageCoordinatorService().getMostRecentImageCoordinator().waitForFITSFiles(timeout);
                });
    }

    /**
     * Wait until all images have been received from the DAQ (but not
     * necessarily written as FITS files)
     *
     * @param timeout The maximum duration to wait
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for image data to arrive", level = Command.NORMAL, autoAck = false)
    public void waitForImages(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> subsys.getImageCoordinatorService().getMostRecentImageCoordinator().waitForImages(timeout));
    }

    /**
     * Wait until the sequencer stops.
     *
     * @param timeout
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for sequencer to stop", level = Command.NORMAL, autoAck = false)
    public void waitForSequencer(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                // Note: dont use waitForStop, because that can return before the state 
                // has been updated, causing subsequent commands to fail
                .action(()
                        -> subsys.getAgentService(AgentStateService.class).waitFor((StateBundle t)
                        -> t.isInState(FocalPlaneState.QUIESCENT) || t.isInState(FocalPlaneState.NEEDS_CLEAR),
                        timeout.toMillis(), TimeUnit.MILLISECONDS));
    }

    /**
     * Adds meta-data to be associated with the next image. This meta-data can
     * potentially be stored in the image database, and be sent to the image
     * handlers for inclusion in the FITS headers. This command is called by the
     * MCM and bot_data scripts, so must be a NORMAL mode command.
     *
     * @param headersMap
     */
    @Command(type = Command.CommandType.ACTION, description = "Set a header keyword value shared by all extensions", level = Command.NORMAL, autoAck = false)
    public void setHeaderKeywords(Map<String, Serializable> headersMap) {
        subsys.helper()
                .enterFaultOnException(true)
                .action(() -> subsys.setHeaderKeywords(headersMap));

    }

    /**
     * Get the full path of the loaded sequencer files.
     *
     * @return A Map containing the reb path - sequencer path
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the Map of reb path, sequencer file full path", level = Command.NORMAL)
    public Map<SequencerType, String> getSequencerPaths() {
        return subsys.getSequencers().getSequencerPaths();
    }

    /**
     * Temporary command for testing alerts
     */
    @Command(type = Command.CommandType.ACTION, description = "Raise badPixels alert")
    public void raiseBadPixelAlert() {
        AlertService alertService = subsys.getAgentService(AlertService.class);
        final String message = String.format("Image %s had %d bad pixels", "TEST-IMAGE",  99999);
        LOG.log(Level.WARNING, message);
        alertService.raiseAlert(ImageCoordinatorService.badPixelAlert, AlertState.ALARM, message);
    }
}
