package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Emulator;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.Playlist;

/**
 * A CCS implementation of a playlist on top of DAQ single image playlists
 * @author tonyj
 */
class CCSPlayList {

    private static final Logger LOG = Logger.getLogger(CCSPlayList.class.getName());

    private final String name;
    private final List<Image> imageList;
    private final boolean repeat;
    private int index;
    private final Emulator emulator;
    private final Path tempFile;

    CCSPlayList(String playListFileName, List<Image> images, boolean repeat, Emulator emulator) throws IOException {
        this.name = playListFileName;
        this.imageList = images;
        this.repeat = repeat;
        this.index = 0;
        this.emulator = emulator;
        tempFile = Files.createTempFile("play", "list");
    }

    String getName() {
        return name;
    }

    boolean hasNextImage() {
        return index < imageList.size();
    }

    /**
     * Get the next image in the play list, without incrementing
     * @return The next image, or null of there is no next image
     */
    Image peekNextImage() {
        return (hasNextImage()) ? imageList.get(index) : null;
    }

    void playNextImage() throws IOException, DAQException {
        if (!hasNextImage()) {
            throw new IOException("Playlist exhausted "+this);
        }
        Image currentImage = imageList.get(index++);
        if (index >= imageList.size() && repeat) {
            index = 0;
        }
        Files.deleteIfExists(tempFile);
        try (Playlist oneImagePlaylist = emulator.openPlaylist(tempFile.toFile())) {
            LOG.log(Level.INFO, "Created temp playlist {0} with image {1}", new Object[]{tempFile, currentImage.getMetaData().getName()});
            oneImagePlaylist.add(currentImage);
            List<Image> images = oneImagePlaylist.getImages();
            LOG.log(Level.INFO, "Starting temp playlist {0} with image list {1}", new Object[]{tempFile, images});
            oneImagePlaylist.start(false);
        }
    }

    @Override
    public String toString() {
        Image next = peekNextImage();
        return "CCSPlayList{" + "name=" + name + ", index=" + index + ", size=" + imageList.size() + ", repeat=" + repeat + ", nextImage=" + (next==null ? "None" : peekNextImage().getMetaData().getName()) + '}';
    }

}
