package org.lsst.ccs.subsystem.focalplane;

import java.time.Duration;
import java.util.Collections;
import java.util.Set;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.focalplane.states.SequencerState;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Focal Plane level commands. These are normal level commands required by
 * LSE-71.
 *
 * @author The LSST CCS Team
 */
public class LSE71Commands {

    private static final LocationSet ALL_LOCATIONS = new LocationSet();
    private static final int MAX_CLEARS = 15; // TODO: This should be configurable

    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig sequencerConfig;

    public enum ReadoutMode {
        TRUE, FALSE, PSEUDO
    };

    LSE71Commands(FocalPlaneSubsystem subsys, SequencerConfig config) {
        this.subsys = subsys;
        this.sequencerConfig = config;
    }

    // Top level commands corresponding to LSE-71
    /**
     * Performs a clear operation. This method returns immediately and does not
     * wait for the sequencer to stop.
     *
     * @param nClears
     */
    @Command(type = Command.CommandType.ACTION, description = "Perform a clear operation", level = Command.NORMAL, autoAck = false)
    public void clear(@Argument(defaultValue = "1") int nClears) {
        subsys.helper()
                .precondition(nClears > 0 && nClears <= MAX_CLEARS, "Invalid nClears: %d", nClears)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    if (subsys.isInState(SequencerState.IDLE_FLUSH)) {
                        // Stop IDLE_FLUSH but do not change state (since we are going to clearing)
                        subsys.getSequencers().endIdleFlush(null);
                    }
                    subsys.getSequencers().clear(nClears);
                });
    }

    /**
     * Allocate an imageName to be used with a future call to
     * startNamedIntegration
     *
     * @return The allocated image name
     */
    @Command(type = Command.CommandType.ACTION, description = "Allocate an image for a future call to startNamedIntegration", level = Command.NORMAL, autoAck = false)
    public ImageName allocateImageName() {
        return subsys.helper()
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .enterFaultOnException(true)
                .action(() -> {
                    return subsys.getImageNameService().getImageName();
                });
    }

    /**
     * Starts an integration operation. This method returns immediately.
     *
     * @param annotation The image annotation
     * @param locations The set of locations to read
     * @return The image name allocated
     */
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public ImageName startIntegration(
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .enterFaultOnException(true)
                .action(() -> {
                    ImageName in = subsys.getImageNameService().getImageName();
                    endIdleFlushInternal();
                    startIntegrationInternal(in, locations, annotation);
                    return in;
                });
    }

    /**
     * Performs clears and then starts an integration operation for a named image. This method returns
     * once the integration has started.
     *
     * @param nClears The number of clears (may be zero)
     * @param annotation The image annotation
     * @param locations The set of locations to read
     * @return The image name allocated
     */
    @Command(type = Command.CommandType.ACTION, description = "Clear and start integration", level = Command.NORMAL, autoAck = false)
    public ImageName clearAndStartIntegration(
            @Argument(defaultValue = "1") int nClears,
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .precondition(nClears >= 0 && nClears <= MAX_CLEARS, "Invalid nClears: %d", nClears)
                .enterFaultOnException(true)
                .action(() -> {
                    ImageName in = subsys.getImageNameService().getImageName();
                    endIdleFlushInternal();
                    if (nClears > 0) {
                        subsys.getSequencers().clear(nClears, null);
                        subsys.getSequencers().waitForStop(Duration.ofSeconds(1));
                    }
                    startIntegrationInternal(in, locations, annotation);
                    return in;
                });
    }
    
    /**
     * Starts an integration operation for a named image. This method returns
     * immediately.
     *
     * @param imageName The image name to use
     * @param annotation The image annotation
     * @param locations The set of locations to read
     */
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public void startNamedIntegration(
            @Argument ImageName imageName,
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .enterFaultOnException(true)
                .action(() -> {
                    endIdleFlushInternal();
                    startIntegrationInternal(imageName, locations, annotation);
                });

    }

    /**
     * Performs clears and then starts an integration operation for a named image. This method returns
     * once the integration has started.
     *
     * @param imageName The image name to use
     * @param nClears The number of clears (may be zero)
     * @param annotation The image annotation
     * @param locations The set of locations to read
     */
    @Command(type = Command.CommandType.ACTION, description = "Clear and start integration", level = Command.NORMAL, autoAck = false)
    public void clearAndStartNamedIntegration(
            @Argument ImageName imageName,
            @Argument(defaultValue = "1") int nClears,
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(nClears >= 0 && nClears <= MAX_CLEARS, "Invalid nClears: %d", nClears)
                .enterFaultOnException(true)
                .action(() -> {
                    endIdleFlushInternal();
                    if (nClears > 0) {
                        subsys.getSequencers().clear(nClears, null);
                        subsys.getSequencers().waitForStop(Duration.ofSeconds(1));
                    }
                    startIntegrationInternal(imageName, locations, annotation);
                });
    }

    private void endIdleFlushInternal() throws RaftException, REBException, DAQException {
        if (subsys.isInState(SequencerState.IDLE_FLUSH)) {
            subsys.getSequencers().endIdleFlush(null);
        }
    }

    private void startIntegrationInternal(ImageName in, Set locations, String annotation) throws DAQException, REBException, RaftException {
        ImageCoordinator currentImage = subsys.getImageCoordinatorService().createImageCoordinator(in);
        LocationSet localLocations = computeLocations(locations);
        if (localLocations.isEmpty()) {
            localLocations = subsys.getSequencers().getConfiguredLocations();
        }
        if (subsys.isInState(SequencerState.IDLE_FLUSH)) {
            // Stop IDLE_FLUSH but do not change state (since we are going to INTEGRATING)
            subsys.getSequencers().endIdleFlush(null);
        }
        String emulatedImageName = null;
        if (sequencerConfig.hasEmulatedDAQ()) {
            CCSPlayList playList = sequencerConfig.getCurrentPlaylist();
            if (playList == null) {
                throw new RaftException("Using emulated DAQ but no playlist defined");
            }
            if (!playList.hasNextImage()) {
                throw new RaftException("No images left in playlist " + playList.getName());
            }
            emulatedImageName = playList.peekNextImage().getMetaData().getName();
            subsys.setHeaderKeywords(Collections.singletonMap("emulatedImage", emulatedImageName));
        }
        subsys.getSequencers().startIntegration(in, annotation, localLocations);
        currentImage.startIntegration(annotation, localLocations, emulatedImageName);
    }

    @Command(type = Command.CommandType.ACTION, description = "Shift n rows", level = Command.NORMAL, autoAck = false)
    public void shiftNRows(int nRows) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .precondition(!sequencerConfig.useStepAfterIntegrate(), "Shift rows no compatible with useStepAfterIntegrate")
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().rowShift(nRows);
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "End exposure, and optionally read out", level = Command.NORMAL, autoAck = false)
    public void endIntegration(@Argument(defaultValue = "TRUE") ReadoutMode readout) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .enterFaultOnException(true)
                .action(() -> {
                    Image image = subsys.getSequencers().endIntegration(readout);
                    subsys.getImageCoordinatorService().endIntegration(readout, image);
                });
    }

    // These are not really LSE-71 commands, so perhapss they do no really belong here?
    
    /**
     * Starts an IDLE_FLUSH operation. This method returns immediately.
     *
     */
    @Command(type = Command.CommandType.ACTION, description = "Start idle flush", level = Command.NORMAL, autoAck = false)
    public void startIdleFlush() {
        subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().startIdleFlush();
                });
    }

    /**
     * Ends an IDLE_FLUSH operation and performs zero or more clears
     * This command is of limited use, because if idleFlushTimeout is on, the 
     * sequencer will go back to IDLE_FLUSH after performing any requested clears anyway,
     *
     * @param nClears The number of clears to do (may be zero)
     */
    @Command(type = Command.CommandType.ACTION, description = "End idle flush", level = Command.NORMAL, autoAck = false)
    public void endIdleFlush(@Argument(defaultValue = "1") int nClears) {
        subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .precondition(SequencerState.IDLE_FLUSH)
                .precondition(nClears >= 0 && nClears <= MAX_CLEARS, "Invalid nClears: %d", nClears)
                .enterFaultOnException(true)
                .action(() -> {
                    if (nClears == 0) {
                        subsys.getSequencers().endIdleFlush(FocalPlaneState.QUIESCENT);
                    } else {
                        // Note, transitions from IDLE_FLUSH -> CLEARING, with no intermediate QUIESCENT state
                        subsys.getSequencers().endIdleFlush(null);
                        subsys.getSequencers().clear(nClears);                    
                    }
                });
    }

    private LocationSet computeLocations(Set locations) {
        if (locations == null || locations.isEmpty()) {
            return ALL_LOCATIONS; // Note, REBDriver interprets empty set to mean "all"
        } else if (locations instanceof LocationSet) {
            return (LocationSet) locations;
        } else {
            LocationSet result = new LocationSet();
            for (Object o : locations) {
                if (o instanceof Location) {
                    result.add((Location) o);
                } else if (o != null) {
                    // Note, we use LocationSet.of to support Strings like R22, as opposed to 
                    // individial locations such as R22/Reb1
                    result.addAll(LocationSet.of(o.toString()));
                }
            }
            return result;
        }
    }
}
