package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;

/**
 * A simple class for calling a set of webhooks when a new event is generated.
 *
 * @author tonyj
 */
class WebHooks {

    private final WebHooksConfig config;
    private final ExecutorService executor;
    private static final Logger LOG = Logger.getLogger(WebHooks.class.getName());
    private static final Duration connectTimeout = Duration.ofSeconds(10);

    WebHooks(ExecutorService executor, WebHooksConfig config) {
        this.config = config;
        this.executor = executor;
    }

    Future<List<Integer>> notifyNewImage(ImageName imageName) {
        return executor.submit(() -> {
            List<Integer> result = new ArrayList<>();
            config.getURLS().forEach(url -> {
                try {
                    URLConnection conn = url.openConnection();
                    if (conn instanceof HttpURLConnection) {
                        HttpURLConnection http = (HttpURLConnection) conn;
                        http.setDoOutput(true);
                        http.addRequestProperty("Content-Type", "application/json");
                        http.addRequestProperty("User-Agent", "CCS focal-plane subsystem");
                        http.setRequestMethod("POST");
                        http.setConnectTimeout((int) connectTimeout.toMillis());
                        http.setReadTimeout((int) connectTimeout.toMillis());
                        http.connect();
                        try {
                            try (OutputStream out = http.getOutputStream()) {
                                final String json = "{\"image\":\"" + imageName + "\"}";
                                out.write(json.getBytes(StandardCharsets.UTF_8));
                            }
                            http.getInputStream().close();
                            result.add(http.getResponseCode());
                        } finally {
                            http.disconnect();
                        }
                    } else {
                        throw new IOException("Invalid URL: " + url);
                    }
                } catch (IOException x) {
                    LOG.log(Level.WARNING, x, () -> String.format("Failed to invoke webhook at %s", url));
                    result.add(-99);
                }
            });
            return result;
        });
    }
}
