package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;
import java.time.Instant;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 *
 * @author tonyj
 */
public class ImageDatabase implements AutoCloseable {

    private final static int MAX_RETRIES = 1;

    private final String dbURL;
    private Connection conn;
    private PreparedStatement stmt1;
    private static final Logger LOG = Logger.getLogger(ImageDatabase.class.getName());
    final ImageDAO dao;

    private static class ImageDAO {

        private ImageName imageName;
        private String imageType;
        private String testType;
        private String runNumber;
        private Integer tSeqNum;
        private final String testStand = "BOT"; // FIXME:
        private String fileLocation;
        private Instant obsDate;
        private final LocationSet locations = new LocationSet();
        private Float exposureTime;
        private Float darkTime;
        private Long imageTag;

        void addMetaData(Map<String, Serializable> metaData) {
            metaData.forEach((name, value) -> {
                switch (name) {
                    case "DarkTime":
                        darkTime = toFloat(value);
                        break;
                    case "ExposureTime":
                        exposureTime = toFloat(value);
                        break;
                    case "TestSeqNum":
                        tSeqNum = toInt(value);
                        break;
                    case "TestType":
                        testType = toString(value);
                        break;
                    case "ImageType":
                        imageType = toString(value);
                        break;
                    case "RunNumber":
                        runNumber = toString(value);
                        break;
                }
            });
        }

        private Float toFloat(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Float) {
                return (Float) value;
            } else if (value instanceof Number) {
                return ((Number) value).floatValue();
            } else {
                return Float.parseFloat(value.toString());
            }
        }

        private Integer toInt(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Integer) {
                return (Integer) value;
            } else if (value instanceof Number) {
                return ((Number) value).intValue();
            } else {
                return Integer.parseInt(value.toString());
            }
        }

        private String toString(Object value) {
            if (value == null) {
                return null;
            } else {
                return value.toString();
            }
        }

        private Date toDate(Object value) {
            if (value == null) {
                return null;
            } else if (value instanceof Date) {
                return ((Date) value);
            } else if (value instanceof Instant) {
                return java.util.Date.from((Instant) value);
            } else {
                throw new RuntimeException("Cannot convert " + value + " to date");
            }
        }

        private void setMetaData(ImageMetaData imageMetaData) {
            obsDate = imageMetaData.getTimestamp();
            imageTag = imageMetaData.getId();
        }

        private void addLocations(LocationSet locations) {
            this.locations.addAll(locations);
        }

        private void setCurrentImage(ImageName currentImage) {
            this.imageName = currentImage;
        }

        private void clear() {
            imageName = null;
            locations.clear();
            imageType = null;
            testType = null;
            runNumber = null;
            tSeqNum = null;
            fileLocation = null;
            obsDate = null;
            exposureTime = null;
            darkTime = null;
            imageTag = null;
        }

    }

    ImageDatabase(String dbURL) {
        this.dbURL = dbURL;
        this.dao = new ImageDAO();
    }

    void ingest(ImageName currentImage, FileList fitsFiles, AbortableCountDownLatch fitsFileCountDown) {
        dao.clear();
        dao.setCurrentImage(currentImage);
        Runnable runnable = () -> {
            try {
                fitsFileCountDown.await();
                dao.fileLocation = fitsFiles.getCommonParentDirectory().toString();
                insertImage();
            } catch (InterruptedException | SQLException ex) {
                LOG.log(Level.WARNING, "Did not commit to database due to error", ex);
            } finally {
                dao.clear();
            }
        };
        // FIXME: Do something better
        Thread t = new Thread(runnable, "Database Ingest Thread");
        t.start();
    }

    void addLocations(LocationSet locations) {
        dao.addLocations(locations);
    }

    void setMetaData(ImageMetaData imageMetaData) {
        dao.setMetaData(imageMetaData);
    }

    void addMetaData(Map<String, Serializable> headersMap) {
        dao.addMetaData(headersMap);
    }

    static int raftsMaskFromLocations(LocationSet locations) {
        int raftMask = 0;
        for (Location location : locations) {
            raftMask |= 1 << (location.index() / 4);
        }
        return raftMask;
    }

    private void openConnection() throws SQLException {
        this.conn = DriverManager.getConnection(dbURL);
        conn.setAutoCommit(false);
        conn.setTransactionIsolation(Connection.TRANSACTION_SERIALIZABLE);

        try (PreparedStatement stmt = conn.prepareStatement(
                "create table if not exists ccs_image ("
                + "telCode varchar(2) not null, "
                + "seqnum integer not null, "
                + "dayobs varchar(8) not null, "
                + "controller varchar(2) not null, "
                + "darkTime float, "
                + "exposureTime float, "
                + "fileLocation varchar(255), "
                + "imageTag bigint, "
                + "imgType varchar(20), "
                + "obsDate timestamp, "
                + "raftMask integer, "
                + "runNumber varchar(20), "
                + "testType varchar(20), "
                + "tseqnum integer, "
                + "tstand varchar(20), "
                + "primary key (telCode, seqnum, dayobs, controller))")) {
            stmt.execute();
        }

        stmt1 = conn.prepareStatement(
                "insert into ccs_image (telcode, seqnum, dayobs, controller, darkTime, exposureTime, fileLocation,"
                + "imageTag, imgType, obsDate, raftMask, runNumber, testType, tseqnum, tstand) "
                + "values (?, ?, ?, ?, ?,"
                + "        ?, ?, ?, ?, ?, "
                + "        ?, ?, ?, ?, ?) ");
    }

    void insertImage() throws SQLException {
        for (int i = 0;; i++) {
            if (conn == null || conn.isClosed()) {
                openConnection();
            }
            try {
                stmt1.setString(1, dao.imageName.getSource().getCode());
                stmt1.setInt(2, dao.imageName.getNumber());
                stmt1.setString(3, dao.imageName.getDateString());
                stmt1.setString(4, dao.imageName.getController().getCode());
                stmt1.setObject(5, dao.darkTime);
                stmt1.setObject(6, dao.exposureTime);
                stmt1.setString(7, dao.fileLocation);
                stmt1.setObject(8, dao.imageTag);
                stmt1.setString(9, dao.imageType);
                stmt1.setObject(10, dao.obsDate != null ? Timestamp.from(dao.obsDate) : null);
                stmt1.setInt(11, raftsMaskFromLocations(dao.locations));
                stmt1.setString(12, dao.runNumber);
                stmt1.setString(13, dao.testType);
                stmt1.setObject(14, dao.tSeqNum);
                stmt1.setString(15, dao.testStand);
                stmt1.executeUpdate();
                conn.commit();
                break;
            } catch (SQLException x) {
                if (i < MAX_RETRIES) {
                    LOG.log(Level.FINE, "Database update failed, retrying", x);
                    try {
                        close();
                    } catch (SQLException ex) {
                        // Ignored, we still want to retry
                    }
                } else {
                    throw x;
                }
            }
        }
    }

    @Override
    public void close() throws SQLException {
        if (conn != null) {
            conn.close();
            this.conn = null;
        }
    }
}
