package org.lsst.ccs.subsystem.focalplane;

import java.util.Map;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.focalplane.data.FormattedHashMap;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.REBDevice;

/**
 * Convenience commands which can be issued at the focal-plane level. Typically
 * used to send query commands to the REBs and tabulate the results.
 *
 * @author tonyj
 */
public class FocalPlaneCommands {

    private final FocalPlaneSubsystem subsys;

    FocalPlaneCommands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    /**
     * Get the CCDType for each REB.
     *
     * @return A map of REBName=>CCDType for each online REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get configured CCD types", level = Command.NORMAL, autoAck = false)
    public Map<String, String> getCCDType() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(reb -> reb.getCcdType().getName());
                });
    }

    /**
     * Gets the value of a named sequencer parameter.
     *
     * @param name The parameter name
     * @return The map of values, one per REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get sequencer parameter values")
    public Map<String, Number> getSequencerParameter(@Argument(name = "name", description = "The parameter name") String name) {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(exceptionCatcher(reb -> reb.getSequencer().getParameter(name)));
                });
    }

    /**
     * Gets the value of a all sequencer parameters.
     *
     * @return The map of values, one per REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get sequencer parameter values")
    public Map<String, Map<String, Integer>> getSequencerParameters() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb((reb) -> {
                        Map<String, Integer> collect = reb.getSequencer().getPointers().entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().getValue()));
                        return collect;
                    });
                });
    }

    /**
     * Gets the list of REB hardware versions.
     *
     * @return The map of REB hardware versions
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of REB hardware versions")
    public Map<String, Number> getREBHwVersions() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(reb -> reb.getHwVersion(), "0x%08x");
                });
    }

    /**
     * Gets the list of REB serial numbers.
     *
     * @return The map of REB serial numbers
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of REB serial numbers")
    public Map<String, Number> getREBSerialNumbers() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(reb -> reb.getSerialNumber(), "%1$d (0x%1$08x)");
                });
    }

    /**
     * Gets the back bias on states.
     *
     * @return A map of the back bias on states for each REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the back bias on states")
    public Map<String, Boolean> isBackBiasOn() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(exceptionCatcher(reb -> reb.isBackBiasOn()));
                });
    }

    /**
     * Gets the running state of all sequencers.
     *
     * @return A map of the sequencer running states for each REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the running states of the sequencers")
    public Map<String, Boolean> isSequencerRunning() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(exceptionCatcher(reb -> reb.getSequencer().isRunning()));
                });
    }

    /**
     * Gets the scan mode for each REB.
     *
     * @return A list of the scan mode states for each REB
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the scan mode enabled states")
    public Map<String, Boolean> isScanEnabled() {
         return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    return forEachReb(exceptionCatcher(reb -> reb.getSequencer().isScanEnabled()));
                });
    }

    private <T extends Object> Map<String, T> forEachReb(Function<REBDevice, T> function) {
        return forEachReb(function, null);
    }

    private <T extends Object> Map<String, T> forEachReb(Function<REBDevice, T> function, String format) {
        return subsys.getRebDevices()
                .entrySet()
                .stream()
                .filter(e -> e.getValue().isOnline())
                .collect(Collectors.toMap(Map.Entry::getKey, e -> function.apply(e.getValue()), throwingMerger(), () -> new FormattedHashMap<>(format)));
    }

    <T, R> Function<T, R> exceptionCatcher(ExceptionThrowingFunction<T, R> function) {
        return (T t) -> {
            try {
                return function.apply(t);
            } catch (Exception x) {
                return (R) x;
            }
        };
    }

    private interface ExceptionThrowingFunction<T, R> {

        R apply(T t) throws Exception;
    }

    private static <T> BinaryOperator<T> throwingMerger() {
        return (u, v) -> {
            throw new IllegalStateException(String.format("Duplicate key %s", u));
        };
    }

}
