package org.lsst.ccs.subsystem.focalplane;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;

/**
 * Commands that are useful when running the focal plane subsystem with
 * scripting
 *
 * @author tonyj
 */
public class ScriptingCommands {

    private final FocalPlaneSubsystem subsys;

    private static final Logger LOG = Logger.getLogger(ScriptingCommands.class.getName());

    ScriptingCommands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    /**
     * Wait until all FITS files have been written.
     *
     * @param timeout The maximum duration to wait
     * @return The list of files which were written
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for FITS files to be saved", level = Command.NORMAL, autoAck = false)
    public FileList waitForFitsFiles(@Argument(defaultValue = "PT30S") Duration timeout) {
        return subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> {
                    return subsys.getImageMessageHandling().waitForFITSFiles(timeout);
                });
    }

    /**
     * Wait until all images have been received from the DAQ (but not
     * necessarily written as FITS files)
     *
     * @param timeout The maximum duration to wait
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for image data to arrive", level = Command.NORMAL, autoAck = false)
    public void waitForImages(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> subsys.getImageMessageHandling().waitForImages(timeout));
    }

    /**
     * Wait until the sequencer stops.
     *
     * @param timeout
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for sequencer to stop", level = Command.NORMAL, autoAck = false)
    public void waitForSequencer(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                // Note: dont use waitForStop, because that can return before the state 
                // has been updated, causing subsequent commands to fail
                .action(()
                        -> subsys.getAgentService(AgentStateService.class).waitFor((StateBundle t)
                        -> t.isInState(FocalPlaneState.QUIESCENT) || t.isInState(FocalPlaneState.NEEDS_CLEAR),
                        timeout.toMillis(), TimeUnit.MILLISECONDS));
    }
}
