/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.FocalPlaneConfig;
import org.lsst.ccs.subsystem.focalplane.FocalPlaneSubsystem;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.location.LocationSet;

public class Sequencers {
    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());
    private static final String CLEAR_MAIN = "Clear";
    private static final String INTEGRATE_MAIN = "Integrate";
    private static final String ROW_SHIFT_R_MAIN = "RowShiftR";
    private static final String ROW_SHIFT_F_MAIN = "RowShiftF";
    private static final String READ_MAIN = "Read";
    private static final String PSEUDO_READ_MAIN = "PsuedoRead";
    private static final String[] MAINS = new String[]{"Clear", "Integrate", "RowShiftR", "RowShiftF", "Read"};
    private static final String CLEAR_COUNT_PARAMETER = "ClearCount";
    private static final String SHIFT_COUNT_PARAMETER = "ShiftCount";
    private static final String[] PARAMETERS = new String[]{"ClearCount", "ShiftCount"};
    private static final Map<Reb.RebType, Integer> REB_TYPE_MAP = new HashMap<Reb.RebType, Integer>();
    private final Map<Reb, SequencerProc> REBs = new LinkedHashMap<Reb, SequencerProc>();
    private String imageName;
    private final FocalPlaneSubsystem subsys;
    private final FocalPlaneConfig focalPlaneConfig;
    private Map<SequencerType, ModelAndName> models;
    private Map<Reb.RebType, int[]> registerMap;
    private KeyValueDataList sequencerKeyValueData;
    private String annotation;
    private LocationSet locations;

    Sequencers(FocalPlaneSubsystem subsys, FocalPlaneConfig focalPlaneConfig) {
        this.subsys = subsys;
        this.focalPlaneConfig = focalPlaneConfig;
    }

    void add(Reb reb, SequencerProc seq) {
        this.REBs.put(reb, seq);
    }

    public void clear(int nClears) throws REBException, RaftException {
        this.setMain(CLEAR_MAIN);
        this.setParameter(CLEAR_COUNT_PARAMETER, nClears);
        this.run(false, FocalPlaneState.CLEARING);
    }

    public void startIntegration(String imageName, String annotation, LocationSet locations) throws REBException, RaftException {
        this.imageName = imageName;
        this.annotation = annotation;
        this.locations = locations;
        this.setMain(INTEGRATE_MAIN);
        this.run(false, FocalPlaneState.INTEGRATING);
    }

    public void rowShift(int nRows) throws REBException, RaftException {
        if (nRows == 0) {
            return;
        }
        if (nRows > 0) {
            this.setMain(ROW_SHIFT_F_MAIN);
        } else {
            this.setMain(ROW_SHIFT_R_MAIN);
        }
        this.setParameter(SHIFT_COUNT_PARAMETER, Math.abs(nRows));
        this.run(false, FocalPlaneState.ROW_SHIFT);
    }

    public void stop() throws REBException {
        for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
            if (!this.isRebOnlineAndValid(entry.getKey())) continue;
            entry.getValue().sendStop();
        }
    }

    public void endIntegration(LSE71Commands.READOUT_MODE readout) throws REBException, RaftException {
        this.stop();
        this.waitForStop(Duration.ofSeconds(1L));
        switch (readout) {
            case TRUE: {
                this.setMain(READ_MAIN);
                this.run(true, FocalPlaneState.READING_OUT);
                break;
            }
            case PSEUDO: {
                this.setMain(PSEUDO_READ_MAIN);
                this.run(false, FocalPlaneState.READING_OUT);
                break;
            }
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
            if (!this.isRebOnlineAndValid(entry.getKey())) continue;
            entry.getValue().setStart(mainName);
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
            if (!this.isRebOnlineAndValid(entry.getKey())) continue;
            entry.getValue().setParameter(parameter, value);
        }
    }

    private void run(boolean acquire, FocalPlaneState stateOnSuccess) throws REBException {
        if (acquire) {
            this.subsys.getGlobalProc().acquireImage(this.imageName, this.focalPlaneConfig.getDAQFolder(), 1, this.annotation, this.locations == null ? new int[]{} : this.locations.asIntArray());
        } else {
            for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
                if (!this.isRebOnlineAndValid(entry.getKey())) continue;
                entry.getValue().startSequencer();
            }
        }
        this.setState(stateOnSuccess);
        this.doInBackground(() -> {
            try {
                this.waitForStop(Duration.ofMillis(1000000000L));
                this.setState(FocalPlaneState.QUIESCENT);
                LOG.info("Sequencers finished");
            }
            catch (REBException ex) {
                LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
            }
        });
    }

    public void waitForStop(Duration timeout) throws REBException {
        for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
            if (!this.isRebOnlineAndValid(entry.getKey())) continue;
            SequencerProc seq = entry.getValue();
            seq.waitDone((int)timeout.toMillis());
            int errorAddr = seq.getErrorAddr();
            if (errorAddr == -1) continue;
            LOG.log(Level.SEVERE, "REB error register set to {0} for {1}", new Object[]{errorAddr, entry.getKey()});
        }
    }

    private static void sanityCheck(FPGA2Model model) throws RaftException, REBException {
        Map mainMap = model.getMainAddresses();
        for (String main : MAINS) {
            if (mainMap.get(main) != null) continue;
            throw new RaftException("Sequencer failed sanity check, missing main " + main);
        }
        List pointers = model.getPointers();
        for (String parameter : PARAMETERS) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (!pointer.getName().equals(parameter)) continue;
                found = true;
                break;
            }
            if (found) continue;
            throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
        }
    }

    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map pointerMap = model.getPointerMap();
        int k = 0;
        block4: for (String metaName : metaDataNames) {
            FPGA2Model.PointerInfo pi = (FPGA2Model.PointerInfo)pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION: {
                        registers[k++] = pi.getAddress();
                        continue block4;
                    }
                    case REPEAT_SUBROUTINE: {
                        registers[k++] = pi.getAddress();
                        continue block4;
                    }
                    default: {
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());
                    }
                }
            }
            throw new RaftException("Required parameter not defined " + metaName);
        }
        LOG.log(Level.INFO, "Meta-data registers set {0}", Arrays.toString(registers));
        return registers;
    }

    static FPGA2Model validate(String sequencerFile) throws RaftException, REBException, IOException {
        InputStream input = BootstrapResourceUtils.getBootstrapResource((String)sequencerFile);
        if (input == null) {
            throw new IOException("Invalid sequencer file name " + sequencerFile);
        }
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        FPGA2Model compiledModel = builder.compileFile(input);
        Sequencers.sanityCheck(compiledModel);
        return compiledModel;
    }

    void load() throws RaftException, REBException {
        LinkedHashMap<SequencerType, ModelAndName> localModels = new LinkedHashMap<SequencerType, ModelAndName>();
        LinkedHashMap<Reb.RebType, int[]> localRegisterMap = new LinkedHashMap<Reb.RebType, int[]>();
        Map<SequencerType, String> sequencers = this.focalPlaneConfig.getSequencers();
        for (Map.Entry<SequencerType, String> sequencer : sequencers.entrySet()) {
            try {
                FPGA2Model fPGA2Model = Sequencers.validate(sequencer.getValue());
                localModels.put(sequencer.getKey(), new ModelAndName(fPGA2Model, sequencer.getValue()));
            }
            catch (IOException iOException) {
                RaftException re = new RaftException("Error reading sequencer " + sequencer.getValue());
                re.initCause((Throwable)iOException);
                throw re;
            }
        }
        KeyValueDataList data = new KeyValueDataList();
        for (Map.Entry<Reb, SequencerProc> entry : this.REBs.entrySet()) {
            Reb reb = entry.getKey();
            SequencerProc seq = entry.getValue();
            ModelAndName modelAndName = Sequencers.getModelAndNameForREB(localModels, reb);
            int[] registers = Sequencers.computeMetaDataRegisters(this.focalPlaneConfig.getMetaDataRegisters(), modelAndName.getModel());
            if (localRegisterMap.get(reb.getRebType()) == null) {
                localRegisterMap.put(reb.getRebType(), registers);
            } else if (!Arrays.equals((int[])localRegisterMap.get(reb.getRebType()), registers)) {
                throw new RaftException("Sequencers have inconsistent meta-data register sets for type " + reb.getRebType());
            }
            if (this.isRebOnlineAndValid(reb)) {
                try {
                    if (seq.isRunning()) {
                        LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
                        seq.sendStop();
                        seq.waitDone(2000);
                    }
                    LOG.log(Level.INFO, "Loading sequencer {0} into {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
                    seq.loadSequencer(modelAndName.getModel());
                }
                catch (REBException x) {
                    LOG.log(Level.SEVERE, "Error stopping or loading sequencer " + modelAndName.getName() + " into " + reb.getFullName(), x);
                }
            }
            data.addData(reb.getFullName() + "/sequencerFile", (Serializable)((Object)modelAndName.getName()));
            data.addData(reb.getFullName() + "/sequencerChecksum", (Serializable)((Object)String.valueOf(modelAndName.getModel().computeCheckSum())));
        }
        this.models = localModels;
        this.registerMap = localRegisterMap;
        this.sequencerKeyValueData = data;
        if (this.subsys != null) {
            for (Map.Entry<Object, Object> entry : localRegisterMap.entrySet()) {
                this.subsys.getGlobalProc().setRegisters(REB_TYPE_MAP.get(entry.getKey()).intValue(), (int[])entry.getValue());
            }
            this.subsys.publishSubsystemDataOnStatusBus((KeyValueData)data);
        }
    }

    private void doInBackground(Runnable run) {
        Thread t = new Thread(run);
        t.setName("Sequencer wait thread");
        t.start();
    }

    private void setState(FocalPlaneState focalPlaneState) {
        if (this.subsys != null) {
            ((AgentStateService)this.subsys.getAgentService(AgentStateService.class)).updateAgentState(new Enum[]{focalPlaneState});
        }
    }

    Map<SequencerType, ModelAndName> getModels() {
        return this.models;
    }

    Map<Reb.RebType, int[]> getRegisterMap() {
        return this.registerMap;
    }

    KeyValueDataList getSequencerKeyValueData() {
        return this.sequencerKeyValueData;
    }

    void load(Reb reb) {
        if (this.models != null) {
            SequencerProc seq = this.REBs.get(reb);
            ModelAndName modelAndName = Sequencers.getModelAndNameForREB(this.models, reb);
            try {
                if (seq.isRunning()) {
                    LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
                    seq.sendStop();
                    seq.waitDone(2000);
                }
                seq.loadSequencer(modelAndName.getModel());
                LOG.log(Level.INFO, "Loading sequencer {0} into REB {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
            }
            catch (REBException ex) {
                LOG.log(Level.SEVERE, "Unable to load sequencer into " + reb.getFullName(), ex);
            }
        }
    }

    private boolean isRebOnlineAndValid(Reb reb) {
        if (this.subsys == null) {
            return true;
        }
        return this.subsys.isRebOnlineAndValid(reb);
    }

    private static ModelAndName getModelAndNameForREB(Map<SequencerType, ModelAndName> models, Reb reb) {
        String rebTypeString;
        switch (rebTypeString = reb.getRebType() + "-" + reb.getCCDType().getName().toUpperCase()) {
            case "SCIENCE-E2V": {
                ModelAndName result = models.get(SequencerType.SCIENCE_E2V);
                if (result == null) {
                    result = models.get(SequencerType.E2V);
                }
                return result;
            }
            case "SCIENCE-ITL": {
                ModelAndName result = models.get(SequencerType.SCIENCE_ITL);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            }
            case "WAVEFRONT-WF": {
                ModelAndName result = models.get(SequencerType.WAVEFRONT);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            }
            case "GUIDER-ITL": {
                ModelAndName result = models.get(SequencerType.GUIDER);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            }
        }
        throw new IllegalArgumentException("Unknown sequencer type: " + rebTypeString);
    }

    static {
        REB_TYPE_MAP.put(Reb.RebType.SCIENCE, 0);
        REB_TYPE_MAP.put(Reb.RebType.WAVEFRONT, 2);
        REB_TYPE_MAP.put(Reb.RebType.GUIDER, 1);
    }

    private static class ModelAndName {
        private final FPGA2Model model;
        private final String name;

        public ModelAndName(FPGA2Model model, String name) {
            this.model = model;
            this.name = name;
        }

        public FPGA2Model getModel() {
            return this.model;
        }

        public String getName() {
            return this.name;
        }
    }
}

