package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.drivers.reb.BaseSet.REB_TYPE_SCIENCE;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.READOUT_MODE;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 * Encapsulates the behavior of all Sequencers in the focal plane. Currently
 * ignores difference between science rafts and wavefront sensors. Currently
 * ignores guiders.
 *
 * @author tonyj
 */
public class Sequencers {

    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());

    //TODO: Should these names be read from configuration?
    private static final String CLEAR_MAIN = "Clear";
    private static final String INTEGRATE_MAIN = "Integrate";
    private static final String ROW_SHIFT_R_MAIN = "RowShiftR";
    private static final String ROW_SHIFT_F_MAIN = "RowShiftF";
    private static final String READ_MAIN = "Read";
    private static final String PSEUDO_READ_MAIN = "PsuedoRead";
    private static final String[] MAINS = {CLEAR_MAIN, INTEGRATE_MAIN, ROW_SHIFT_R_MAIN, ROW_SHIFT_F_MAIN, READ_MAIN};

    private static final String CLEAR_COUNT_PARAMETER = "ClearCount";
    private static final String SHIFT_COUNT_PARAMETER = "ShiftCount";
    private static final String[] PARAMETERS = {CLEAR_COUNT_PARAMETER, SHIFT_COUNT_PARAMETER};

    private final Map<SequencerProc, String> REBs = new LinkedHashMap<>();
    private String imageName;
    private final FocalPlaneSubsystem subsys;
    private final FocalPlaneConfig focalPlaneConfig;
    private FPGA2Model model;
    private String sequencerName;
    private long checksum;
    
    Sequencers(FocalPlaneSubsystem subsys, FocalPlaneConfig focalPlaneConfig) {
        this.subsys = subsys;
        this.focalPlaneConfig = focalPlaneConfig;
    }

    void add(String path, SequencerProc seq) {
        REBs.put(seq, path);
    }

    public void clear(int nClears) throws REBException, RaftException {
        setMain(CLEAR_MAIN);
        setParameter(CLEAR_COUNT_PARAMETER, nClears);
        run(false, FocalPlaneState.READING_OUT);
    }

    public void startIntegration(String imageName) throws REBException, RaftException {
        this.imageName = imageName;
        setMain(INTEGRATE_MAIN);
        run(false, FocalPlaneState.INTEGRATING);
    }

    public void rowShift(int nRows) throws REBException, RaftException {
        if (nRows == 0) {
            return;
        } else if (nRows > 0) {
            setMain(ROW_SHIFT_F_MAIN);
        } else {
            setMain(ROW_SHIFT_R_MAIN);
        }
        setParameter(SHIFT_COUNT_PARAMETER, Math.abs(nRows));
        run(false, FocalPlaneState.ROW_SHIFT);
    }

    public void stop() throws REBException {
        for (SequencerProc reb : REBs.keySet()) {
            reb.sendStop();
        }
    }

    public void endIntegration(READOUT_MODE readout) throws REBException, RaftException {
        stop();
        waitForStop(Duration.ofSeconds(1));
        switch (readout) {
            case TRUE:
                setMain(READ_MAIN);
                run(true, FocalPlaneState.READING_OUT);
                break;
            case PSEUDO:
                setMain(PSEUDO_READ_MAIN);
                run(false, FocalPlaneState.READING_OUT);
                break;
            case FALSE:
            // Nothing to do
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        for (SequencerProc reb : REBs.keySet()) {
            reb.setStart(mainName);
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (SequencerProc reb : REBs.keySet()) {
            reb.setParameter(parameter, value);
        }
    }

    private void run(boolean acquire, FocalPlaneState stateOnSuccess) throws REBException {
        if (acquire) {
            subsys.getGlobalProc().acquireImage(imageName);
        } else {
            for (SequencerProc reb : REBs.keySet()) {
                reb.startSequencer();
            }
        }
        //TODO: What happens if above fails, state should not be left unchanged.
        setState(stateOnSuccess);
        doInBackground(() -> {
            try {
                // No way to tell how long we have to wait for infinite loop, 
                // so wait a somewhat arbitrary 1,000,000 seconds.
                waitForStop(Duration.ofMillis(1_000_000_000));
                setState(FocalPlaneState.QUIESCENT);
                LOG.info("Sequencers finished");
            } catch (REBException ex) {
                LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
            }
        });
    }

    public void waitForStop(Duration timeout) throws REBException {
        for (SequencerProc reb : REBs.keySet()) {
            // NOTE: waitDone throws a REBException it if times out
            reb.waitDone((int) timeout.toMillis());
            int errorAddr = reb.getErrorAddr();
            if (errorAddr != -1) {
                LOG.log(Level.SEVERE, "REB error register set to {0} for {1}", new Object[]{errorAddr, REBs.get(reb)});
                //TODO: We should go into fault state here
            }
        }
    }

    /**
     * Fail fast if the sequencers loaded do not meet our minimal requirements
     *
     * @param seq The sequencer to test
     * @throws RaftException If any of the required mains or parameters are
     * missing
     */
    private static void sanityCheck(FPGA2Model model) throws RaftException, REBException {
        Map<String, Integer> mainMap = model.getMainAddresses();
        for (String main : MAINS) {
            if (mainMap.get(main) == null) {
                throw new RaftException("Sequencer failed sanity check, missing main " + main);
            }
        }
        List<FPGA2Model.PointerInfo> pointers = model.getPointers();
        for (String parameter : PARAMETERS) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (pointer.getName().equals(parameter)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
            }
        }
    }

    /**
     * Check that all required meta-data parameters are present in the
     * sequencer.
     *
     * @param metaData A map of required parameters (TODO: Should be just a
     * list?)
     * @param model The sequencer mode to check
     * @return The list of addresses of the meta-data registers
     * @throws RaftException If one or more meta-data parameters are missing or
     * invalid
     */
    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map<String, PointerInfo> pointerMap = model.getPointerMap();
        int k = 0;
        for (String metaName : metaDataNames) {
            PointerInfo pi = pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION:
                        registers[k++] = pi.getAddress();
                        break;
                    case REPEAT_SUBROUTINE:
                        registers[k++] = pi.getAddress();
                        break;
                    default:
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());

                }
            } else {
                throw new RaftException("Required parameter not defined " + metaName);
            }
        }
        return registers;
    }

    static FPGA2Model validate(String sequencerFile) throws RaftException, REBException, IOException {
        InputStream input = BootstrapResourceUtils.getBootstrapResource(sequencerFile);
        if (input == null) {
            throw new IOException("Invalid sequencer file name " + sequencerFile);
        }
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        FPGA2Model compiledModel = builder.compileFile(input);

        sanityCheck(compiledModel);
        return compiledModel;
    }

    /**
     * Called to load the configured sequencer into all registered
     * SequencerProcs.
     *
     * @throws RaftException
     * @throws REBException
     */
    void load() throws RaftException, REBException {
        try {
            this.model = validate(focalPlaneConfig.getSequencer());
            this.sequencerName = focalPlaneConfig.getSequencer();
            this.checksum = model.computeCheckSum();

            // TODO: The set of meta-data registers should be configurable
            int[] registers = computeMetaDataRegisters(SequencerData.imageRegNames, model);
            // TODO: Probably this is not the right place to do this
            if (subsys != null) {
                subsys.getGlobalProc().setRegisters(REB_TYPE_SCIENCE, registers);
            }

            KeyValueDataList data = new KeyValueDataList();

            AtomicInteger n = new AtomicInteger(0);
            REBs.forEach((sequencer, path) -> {
                try {
                    sequencer.loadSequencer(model);
                    n.incrementAndGet();
                    data.addData(path + "/sequencerFile", sequencerName);
                    data.addData(path + "/sequencerChecksum", String.valueOf(checksum));
                } catch (REBException ex) {
                    LOG.log(Level.WARNING, "Unable to load sequencer " + sequencerName + " into " + path, ex);
                }
            });
            //This check is needed for tests as they provide a null subsys variable.
            if (subsys != null) {
                subsys.publishSubsystemDataOnStatusBus(data);
            }
            LOG.log(Level.INFO, "Loading sequencer {0} into {1} sequencers", new Object[]{focalPlaneConfig.getSequencer(), n});
        } catch (IOException x) {
            RaftException re = new RaftException("Error reading sequencer " + focalPlaneConfig.getSequencer());
            re.initCause(x);
            throw re;
        }
    }

    private void doInBackground(Runnable run) {
        Thread t = new Thread(run);
        t.setName("Sequencer wait thread");
        t.start();
    }

    private void setState(FocalPlaneState focalPlaneState) {
        if (subsys != null) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(focalPlaneState);
        }
    }

    /**
     * Load the compiled sequencer into the specified SequencerProc
     *
     * @param sequencer The SequencerProc to load.
     */
    void load(SequencerProc sequencer) {
        if (model != null) {
            String path = REBs.get(sequencer);
            try {
                KeyValueDataList data = new KeyValueDataList();
                sequencer.loadSequencer(model);
                LOG.log(Level.INFO, "Loading sequencer {0} into REB {1}", new Object[]{sequencerName, path});
                data.addData(path + "/sequencerFile", sequencerName);
                data.addData(path + "/sequencerChecksum", String.valueOf(checksum));
                if (subsys != null) {
                    subsys.publishSubsystemDataOnStatusBus(data);
                }

            } catch (REBException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Unable to load sequencer into " + path, ex);
            }
        }
    }
}
