package org.lsst.ccs.subsystem.focalplane;

import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.reb.Image;
import org.lsst.ccs.subsystem.focalplane.ScriptingCommands.FileList;
import org.lsst.ccs.subsystem.rafts.ImageProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;

/**
 *
 * @author tonyj
 */
class ImageHandling {

    private static final Logger LOG = Logger.getLogger(ImageHandling.class.getName());

    private final FocalPlaneSubsystem subsys;
    private final FocalPlaneConfig focalPlaneConfig;
    private AbortableCountDownLatch fitsFileCountDown;
    private AbortableCountDownLatch imageCountDown;
    private int nImageProc;
    private final FileList fitsFiles = new FileList();

    public ImageHandling(FocalPlaneSubsystem subsys, FocalPlaneConfig focalPlaneConfig) {
        this.subsys = subsys;
        this.focalPlaneConfig = focalPlaneConfig;
    }

    void addImageProc(ImageProc imageProc) {
        imageProc.addImageListener((ImageProc proc, Image image) -> {
            try {
                imageCountDown.countDown();
                if (focalPlaneConfig.isFITSAutoSave()) {
                    proc.setFitsFileNamePattern(focalPlaneConfig.getFITSFilePattern());
                    List<String> files = proc.saveFitsImage(focalPlaneConfig.getFITSRootDirectory() + File.separator + focalPlaneConfig.getFITSDirectoryPattern(), null, true);
                    LOG.log(Level.INFO, "Saved fits files {0}", files);
                    fitsFiles.addAll(files);
                    fitsFileCountDown.countDown();
                }
            } catch (IOException | RaftException ex) {
                LOG.log(Level.SEVERE, "Error saving fits file ", ex);
                imageCountDown.abort(ex);
                fitsFileCountDown.abort(ex);
            }
        });
        nImageProc++;
    }

    void clearWait() {
        imageCountDown = new AbortableCountDownLatch(nImageProc);
        fitsFileCountDown = new AbortableCountDownLatch(nImageProc);
        fitsFiles.clear();
    }

    FileList waitForFITSFiles(Duration timeout) throws InterruptedException, TimeoutException {
        if (fitsFileCountDown.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
            return fitsFiles;
        } else {
            throw new TimeoutException("Timed out while waiting for FITS files");
        }
    }

    void waitForImages(Duration timeout) throws TimeoutException, InterruptedException {
        if (!imageCountDown.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
            throw new TimeoutException("Timed out while waiting for images");
        }
    }

    private static class AbortableCountDownLatch extends CountDownLatch {

        protected Exception abortException = null;

        public AbortableCountDownLatch(int count) {
            super(count);
        }

        /**
         * Unblocks all threads waiting on this latch and cause them to receive
         * an AbortedException. If the latch has already counted all the way
         * down, this method does nothing.
         */
        public void abort(Exception x) {
            if (getCount() == 0) {
                return;
            }

            this.abortException = x;
            while (getCount() > 0) {
                countDown();
            }
        }

        @Override
        public boolean await(long timeout, TimeUnit unit) throws InterruptedException {
            final boolean rtrn = super.await(timeout, unit);
            if (abortException != null) {
                throw new AbortedException(abortException);
            }
            return rtrn;
        }

        @Override
        public void await() throws InterruptedException {
            super.await();
            if (abortException != null) {
                throw new AbortedException(abortException);
            }
        }

        public static class AbortedException extends InterruptedException {

            public AbortedException(Exception cause) {
                super("Aborted due to exception");
                super.initCause(cause);
            }
        }
    }
}
