package org.lsst.ccs.subsystem.focalplane;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.rafts.data.RaftException;

/**
 * Sequencer and Fits File configuration for the focal plane subsystem
 */
@SuppressWarnings("FieldMayBeFinal")
public class FocalPlaneConfig implements ConfigurationBulkChangeHandler {
    
    @ConfigurationParameter(category = "Sequencer")
    private Map<SequencerType, String> sequencer = new HashMap<>();

    @ConfigurationParameter(category = "FitsHandling")
    private String FITSRootDirectory = System.getProperty("java.io.tmpdir");
    
    @ConfigurationParameter(category = "FitsHandling")
    private String FITSDirectoryPattern = "${imageDate}/${imageName}";

    @ConfigurationParameter(category = "FitsHandling")
    private String FITSFilePattern = "${imageName}_${raftName}_S${sensorLoc}.fits";

    @ConfigurationParameter(category = "FitsHandling")
    private boolean FITSAutoSave = true;

    public FocalPlaneConfig() {
        sequencer.put(SequencerType.ITL,"FP_ITL_2s_ir2.seq");
    }
    
    public String getSequencer() {
        return sequencer.get(SequencerType.ITL);
    }

    public String getFITSRootDirectory() {
        return FITSRootDirectory;
    }

    public String getFITSDirectoryPattern() {
        return FITSDirectoryPattern;
    }

    public String getFITSFilePattern() {
        return FITSFilePattern;
    }

    public boolean isFITSAutoSave() {
        return FITSAutoSave;
    }

    void setSequencer(String sequencer) {
        this.sequencer.put(SequencerType.ITL,sequencer);
    }

    @Override
    public void setParameterBulk(Map<String, Object> parametersView) {
    }

    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        String dir = parametersView.get("FITSRootDirectory").toString();
        File dirFile  = new File(dir);
        dirFile.mkdirs();
        if (!dirFile.isDirectory()) throw new IllegalArgumentException("FITSRootDirectory: Not a directory");
        if (!dirFile.canWrite()) throw new IllegalArgumentException("FITSRootDirectory: Not writable");
        // TODO: Validate sequencers in map all exist
        Map<SequencerType,String> sequencerMap = (Map<SequencerType,String>) parametersView.get("sequencer");
        if (sequencerMap.isEmpty()) throw new IllegalArgumentException("Invalid empty sequencer map");
        for (String sequencerFile : sequencerMap.values()) {
            try {
                Sequencers.validate(sequencerFile);
            } catch (IOException | REBException | RaftException ex) {
                throw new IllegalArgumentException("Invalid sequencer file "+sequencerFile, ex);
            }
        }
    }
}
