package org.lsst.ccs.subsystem.focalplane;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsServiceHeaderCommands;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.GlobalProc;
import org.lsst.ccs.subsystem.rafts.ImageProc;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.RaftsCommands;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;

public class FocalPlaneSubsystem extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevices = new HashMap();

    private Sequencers sequencers;
    private final Map<String, RaftsCommands> raftsCommands = new HashMap<>();
    private final FocalPlane focalPlaneGeometry;
    private ImageHandling imageHandling;
    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystem.class.getName());
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalProc globalProc;
        
    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dictionaryService;
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private FocalPlaneConfig focalPlaneConfig;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent agent;

    private ComponentLookup lookupService;
    
    public FocalPlaneSubsystem(FocalPlane geometry) {
        super("fp", AgentInfo.AgentType.WORKER);
        this.focalPlaneGeometry = geometry;
    }

    @Override
    public void build() {
        setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.FOCAL_PLANE.name());
        
        lookupService = agent.getComponentLookup();
        ComponentNode thisComponent = lookupService.getComponentNodeForObject(this);
        for (Raft r : focalPlaneGeometry.getChildrenList()) {

            ComponentNode raftNode = new ComponentNode(r.getName(), new HardwareIdConfiguration());
            lookupService.addComponentNodeToLookup(thisComponent, raftNode);
            for (Reb reb : r.getChildrenList()) {
                String rebName = r.getName() + "/" + reb.getName();
                ComponentNode rebComponent = lookupService.getNodeByPath(rebName);
                for (int j = 0; j < 3; j++) {
                    String sensorName = "S" + reb.getParallelPosition() + j;
                    ComponentNode sensorNode = new ComponentNode(sensorName, new HardwareIdConfiguration());
                    lookupService.addComponentNodeToLookup(rebComponent, sensorNode);
                }
            }
        }
    }

    @Override
    public void postInit() {

        sequencers = new Sequencers(this, focalPlaneConfig);
        imageHandling = new ImageHandling(this, focalPlaneConfig);
        
        updateAgentState(FocalPlaneState.NEEDS_CLEAR);

        for (Raft raft : focalPlaneGeometry.getChildrenList()) {
            String raftName = raft.getName();
            // TODO: Shouldn't the raft know when REB devices belong to it?
            List<REBDevice> raftRebDevices = new ArrayList();
            rebDevices.forEach((rebName, device) -> {
                if (rebName.startsWith(raftName)) {
                    raftRebDevices.add(device);
                    sequencers.add(lookupService.getComponentNodeForObject(device).getPath(),device.getSequencer());
                    final ImageProc imageProc = device.getImageProc();
                    imageProc.setPatternProperty("raftName", raftName);
                    imageHandling.addImageProc(imageProc);
                }
            });

            RaftsCommands commands = new RaftsCommands(this, raftRebDevices, globalProc, raft, null);
            raftsCommands.put(raftName, commands);

            // Add the RaftCommands to the CommandSet
            // TODO: Decide if these commands are actually useful at the focal plane level.
            addCommandsFromObject(commands, raftName);
        }

        agentStateService.addStateChangeListener((Object device, Enum newState, Enum oldState) -> {
            if (newState == RebValidationState.VALID) {
                final REBDevice rebDevice = (REBDevice) device;
                sequencers.load(rebDevice.getSequencer());
                try {
                    Object result = rebDevice.loadAspics(true);
                    LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{rebDevice.getFullName(), result});
                } catch (Exception x) {
                    LOG.log(Level.WARNING,"Exception while loading ASPICS",x);
                }
            }
        }, RebDeviceState.class, RebValidationState.class);

        addCommandsFromObject(new LSE71Commands(this), "");
        addCommandsFromObject(new ScriptingCommands(this),"");
        addCommandsFromObject(new FitsServiceHeaderCommands(this),"");
        
        //globalProc.initialize(rebDevices, clientFactory, sLog);
    }
    
    @Override
    public void preStart() {        
        //Manipulate the dictionary here: 
        for ( DataProviderInfo data: dictionaryService.getDataProviderDictionary().getDataProviderInfos() ) {
            FocalPlaneDataGroup dataGroup = FocalPlaneDataGroup.findFocalPlanDataGroup(data);
            if ( dataGroup != null ) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }

    @Override
    public void postStart() {
        configService.addConfigurationListener(new ConfigListener(this));
        loadSequencers();
    }
    
    void loadSequencers() {
        try {
            sequencers.load();
        } catch (RaftException | REBException x) {
            //TODO: Go into fault state?
            throw new RuntimeException("Error configuring sequencers", x);
        }
    }
    
    void loadASPICS() {
        for (REBDevice device : rebDevices.values()) {
            if (device.isOnline() && device.isSerialNumValid()) {
                try {
                    Object result = device.loadAspics(true);
                    LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{device.getFullName(), result});
                } catch (Exception ex) {
                    //TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error while loading ASPICS", ex);
                }
            }
        }
    }

    public Sequencers getSequencers() {
        return sequencers;
    }

    GlobalProc getGlobalProc() {
        return globalProc;
    }

    ImageNameService getImageNameService() {
        return imageNameService;
    }

    ImageHandling getImageHandling() {
        return imageHandling;
    }
}
