package org.lsst.ccs.subsystem.focalplane;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.Measurement;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.PIDController;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.data.RaftException;

/**
 *  Implements a temperature controller for the rafts system.
 *
 *  @author The LSST CCS Team
 */
public class TempControl implements HasLifecycle {

    private final static String TEMP_CONTROL_CATEGORY = "RaftTempControl";
    private final static String TEMP_CONTROL_STATUS_CATEGORY = "RaftTempControlStatus";
    private static final java.util.logging.Logger LOG = java.util.logging.Logger.getLogger(TempControl.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupName
    private String name;

    @LookupPath
    private String path;
    
    private String raftPath;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String,Channel> allChannels = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    private final Map<String,REBDevice> allRebs = new HashMap<>();
    
    @ConfigurationParameter( category = TEMP_CONTROL_STATUS_CATEGORY, description = "True if loop is active", units="unitless")
    private volatile boolean active = false;

    @ConfigurationParameter( category = TEMP_CONTROL_CATEGORY, description = "maximum PID output", units="Watt")
    private volatile double  maxOutput = 5.6;

    @ConfigurationParameter( category = TEMP_CONTROL_CATEGORY, description = "Temperature set point", units="C" )
    private volatile double setTemp;

    @ConfigurationParameter( maxLength = 9, category = TEMP_CONTROL_CATEGORY, description = "Temperature channels to use", units="unitless")
    private volatile String[] tempChans = new String[] {"Reb0/S01/Temp", "Reb2/S11/Temp", "Reb2/S21/Temp"};

    @ConfigurationParameter( maxLength = 3, category = TEMP_CONTROL_CATEGORY, description = "The REBs with heaters to use", units="unitless" )
    private volatile String[] rebs = new String[] {"Reb0", "Reb2"};

    @ConfigurationParameter(category = TEMP_CONTROL_CATEGORY, description = "input smoothing time", units="second")
    private volatile double  smoothTime = 20.0;

    @ConfigurationParameter(isFinal = true, category = TEMP_CONTROL_CATEGORY, description = "base power input", units = "Watt" )
    private volatile double  basePower = 0.0;

    @ConfigurationParameter(isFinal = true, category = TEMP_CONTROL_CATEGORY, description = "minimum PID output", units="Watt")
    private volatile double  minOutput = 0.0; 

    @ConfigurationParameter(isFinal = true, category = TEMP_CONTROL_CATEGORY, description = "maximum input", units="unitless")
    private volatile double  maxInput = 100.0;

    @ConfigurationParameter(isFinal = true, category = TEMP_CONTROL_CATEGORY, description = "minimum input", units="unitless")
    private volatile double  minInput = -200.0;

    @ConfigurationParameter(category = TEMP_CONTROL_CATEGORY, description = "integration time constant", units="second")
    private volatile double  timeConst = 120.0;

    @ConfigurationParameter(category = TEMP_CONTROL_CATEGORY, description = "PID P coeficient", units="unitless")
    private volatile double  coefP = 3.0;

    @ConfigurationParameter(category = TEMP_CONTROL_CATEGORY, description = "PID I coeficient", units="unitless")
    private volatile double  coefI = 1.0;

    @ConfigurationParameter(category = TEMP_CONTROL_CATEGORY, description = "PID D coeficient", units="unitless")
    private volatile double  coefD = 0.0;

    private PIDController pic;          // The PI controller

    private final List<Channel> tempChansL = new ArrayList<>();  // Temperature channels to use
    private final List<REBDevice> rebsL = new ArrayList<>();     // REBs with heaters to use

    private double lastPower;          // The last power value set

    /**
     *  Start temperature control task.
     */
    @Override
    public void build() {
        if ( !raftPath.endsWith("/") ) {
            raftPath += "/";
        }
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("tempControl/"+raftPath.replace("/", ""), () -> {
            iterateLoop();
        }).withPeriod(Duration.ofMillis(30000)));
    }


    @Override
    public void postInit() {
        pic = new PIDController(raftPath.replace("/", ""));
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setCoefP(coefP);
        pic.setCoefI(coefI);
        pic.setCoefD(coefD);
        pic.setSmoothTime(smoothTime);
        pic.setTimeConst(timeConst);
        pic.setBaseOutput(basePower);

        setTemperatureChannels(tempChans);
        setPowerRebs(rebs);
        
        setTemperature(setTemp);      
        setActive(active);
    }


    @ConfigurationParameterChanger(propertyName = "active")
    public void setActive(boolean active) {
        if ( pic != null ) {
            if ( active ) {
                pic.reset();
            } else {
                setPower(0.0);                
            }
        }
        this.active = active;
    }

    @ConfigurationParameterChanger(propertyName = "maxOutput")
    public void setMaxOutput(double maxOutput) {
        if ( pic != null ) {
            pic.setOutputRange(minOutput, maxOutput);
        }
        this.maxOutput = maxOutput;
    }

    @ConfigurationParameterChanger(propertyName = "timeConst")
    public void setTimeConst(double timeConst) {
        if ( pic != null ) {
            pic.setTimeConst(timeConst);
        }
        this.timeConst = timeConst;
    }

    @ConfigurationParameterChanger(propertyName = "coefP")
    public void setCoefP(double value) {
        this.coefP = value;
        if ( pic != null ) {
            pic.setCoefP(coefP);
        }
    }
        
    @ConfigurationParameterChanger(propertyName = "coefI")
    public void setCoefI(double value) {
        this.coefI = value;
        if ( pic != null ) {
            pic.setCoefI(coefI);
        }
    }
        
    @ConfigurationParameterChanger(propertyName = "coefD")
    public void setCoefD(double value) {
        this.coefD = value;
        if ( pic != null ) {
            pic.setCoefD(coefD);
        }
    }
        
    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     */
    @ConfigurationParameterChanger(propertyName = "setTemp")
    public void setTemperature(double temp) {
        this.setTemp = temp;
        if ( pic != null ) {
            pic.setSetpoint(setTemp);
        }
    }
    
    @ConfigurationParameterChanger(propertyName = "tempChans")
    public void setTemperatureChannels(String[] tempChans) {   
        List<Channel> tmpChansList = new ArrayList<>();
        for (String cName : tempChans) {
            Channel chan = allChannels.get(raftPath+cName);
            if (chan != null) {
                tmpChansList.add(chan);
            } else {                
                throw new RuntimeException("Invalid configuration "+Arrays.asList(tempChans)+". Channel "+cName+" does not exist in "+allChannels);
            }
        }
        tempChansL.clear();
        tempChansL.addAll(tmpChansList);
        this.tempChans = tempChans;
    }


    /**
     * Change the list of rebs used to control the raft temperature.
     * NOTE: when the rebs are changed we need to synchronize as the
     * iterateLoop could be invoked on another thread and reset the heater
     * power while we are removing/adding rebs.
     * @param rebs 
     */
    @ConfigurationParameterChanger(propertyName = "rebs")
    public void setPowerRebs(String[] rebs) {   
        List<REBDevice> tmpRebList = new ArrayList<>();
        for (String cName : rebs) {
            REBDevice reb = allRebs.get(raftPath+cName);
            if (reb != null) {
                tmpRebList.add(reb);
            } else {                
                throw new RuntimeException("Invalid configuration "+Arrays.asList(rebs)+". Reb Device "+raftPath+cName+" does not exist in "+allRebs);
            }
        }
        
        synchronized(rebsL) {
            setPower(0);
            rebsL.clear();
            rebsL.addAll(tmpRebList);
        }
        this.rebs = rebs;
    }    
    
    /**
     *  Sets the heater power.
     *
     *  @param  power  The power to set
     */
    private void setPower(double power) {
        synchronized (rebsL) {
            for (REBDevice reb : rebsL) {
                try {
                    if ( reb.isOnline() ) {
                        reb.setHeaterPower(0, power / rebs.length);
                    }
                } catch (RaftException e) {
                    LOG.log(Level.SEVERE, "Error setting " + reb.getName() + "(isOnline:"+reb.isOnline()+") heater", e);
                }
            }
        }
    }


    /**
     *  Timer method for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        int count = 0;
        double tstamp = 0.0;
        double temp = 0.0;
        for (Channel tempChan : tempChansL) {
            Measurement point = tempChan.getLastMeasurement();
            if (!Double.isNaN(point.getValue())) {
                temp += point.getValue();
                tstamp += point.getCCSTimestamp().getTAIDouble();
                count++;
            }
        }
        if (count > 0) {
            lastPower = pic.performPID(new double[]{temp / count}, tstamp / count);
            setPower(lastPower);
        }
        else {
            LOG.log(Level.SEVERE, "Control loop iteration failed: no valid temperature values available");
        }
    }

}
