package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.daq.ims.Catalog;
import org.lsst.ccs.daq.ims.Emulator;
import org.lsst.ccs.daq.ims.Folder;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.Playlist;
import org.lsst.ccs.daq.ims.Utils;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;

/**
 * Commands that are when using daq emulation to control DAQ playlists
 *
 * @author tonyj
 */
public class PlaylistCommands {

    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig config;

    private static final Logger LOG = Logger.getLogger(PlaylistCommands.class.getName());

    PlaylistCommands(FocalPlaneSubsystem subsys, SequencerConfig config) {
        this.subsys = subsys;
        this.config = config;
    }

    /**
     * Start a playlist. Note this method now works by tracking the play list item inside
     * CCS, and spoon feeding the DAQ single image playlists.
     *
     * @param playListFileName The play list name
     * @param repeat Whether the playlist should be started in repeat mode
     */
    @Command(type = Command.CommandType.ACTION, description = "Start a playlist", level = Command.NORMAL, autoAck = false)

    public void play(String playListFileName, @Argument(defaultValue = "false") boolean repeat) throws IOException {
        Path playListPath = getPlaylistBaseDir(playListFileName);

        subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(config.hasEmulatedDAQ(), "play command only evailable with emulated DAQ")
                .precondition(Files.exists(playListPath), "play list backing file does not exist")
                .action(() -> {
                    final Emulator emulator = config.getEmulator();
                    try (Playlist playList = emulator.openPlaylist(playListPath.toFile())) {
                        CCSPlayList cpl = new CCSPlayList(playListFileName, playList.getImages(), repeat, emulator);
                        config.setCurrentPlaylist(cpl);
                        LOG.log(Level.INFO, "Playlist set {0}", cpl);
                    }
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "Define a playlist", level = Command.NORMAL, autoAck = false)
    public void definePlaylist(String playListFileName, String daqFolder, String... imageNames) {

        subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(config.hasEmulatedDAQ(), "play command only evailable with eumlated DAQ")
                .action(() -> {
                    Path playListPath = getPlaylistBaseDir(playListFileName);
                    if (Files.deleteIfExists(playListPath));
                    Catalog catalog = config.getStore().getCatalog();
                    Folder folder = catalog.find(daqFolder);
                    if (folder == null) {
                        throw new RuntimeException("Specified daq folder does not exist: " + daqFolder);
                    }
                    List<Image> images = new ArrayList<>();
                    for (String imageName : imageNames) {
                        Image image = folder.find(imageName);
                        if (image == null) {
                            throw new RuntimeException("Specified image does not exist: " + imageName);
                        }
                        images.add(image);
                    }

                    try (Playlist playList = config.getEmulator().openPlaylist(playListPath.toFile())) {
                        for (Image image : images) {
                            playList.add(image);
                        }
                    }
                    return null;
                });
    }

    @Command(type = Command.CommandType.QUERY, description = "Show contents of a playlist", level = Command.NORMAL, autoAck = false)
    public String showPlaylist(String playListFileName) throws IOException {
        Path playListPath = getPlaylistBaseDir(playListFileName);

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(config.hasEmulatedDAQ(), "playlists only evailable with eumlated DAQ")
                .precondition(Files.exists(playListPath), "play list backing file does not exist")
                .action(() -> {
                    try (Playlist playList = config.getEmulator().openPlaylist(playListPath.toFile())) {
                        List<Image> images = playList.getImages();
                        return images.stream()
                                .map(image -> String.format("%s %s %s %s", image.getMetaData().getName(), Utils.imageSize(image), image.getMetaData().getTimestamp(), image.getMetaData().getAnnotation()))
                                .collect(Collectors.joining("\n"));
                    }
                });
    }

    @Command(type = Command.CommandType.QUERY, description = "Show status of the current playlist", level = Command.NORMAL, autoAck = false)
    public String showCurrentPlaylist() throws IOException {

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(config.hasEmulatedDAQ(), "playlists only evailable with eumlated DAQ")
                .precondition(config.getCurrentPlaylist() != null, "no playlist currently active")
                .action(() -> {
                    return config.getCurrentPlaylist().toString();
                });
    }

    @Command(type = Command.CommandType.QUERY, description = "Show status of the current playlist", level = Command.NORMAL, autoAck = false)
    public String listPlaylists() throws IOException {

        return subsys.helper()
                .enterFaultOnException(false)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .precondition(config.hasEmulatedDAQ(), "playlists only evailable with eumlated DAQ")
                .action(() -> {
                    List<Path> playlists = Files.list(config.getPlaylistBaseDir()).collect(Collectors.toList());
                    List<String> results = new ArrayList<>();
                    for (Path path : playlists) {
                        try (Playlist playList = config.getEmulator().openPlaylist(path.toFile())) {
                            results.add(String.format("%s: size=%d", path.getFileName(), playList.getImages().size()));
                        }
                    }
                    return results.stream().collect(Collectors.joining("\n"));
                });
    }

    private Path getPlaylistBaseDir(String playlistFileName) throws IOException {
        if (!playlistFileName.endsWith(".playlist")) {
            playlistFileName += ".playlist";
        }
        return config.getPlaylistBaseDir().resolve(playlistFileName);
    }

}
