package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.services.AgentExecutionService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.focalplane.states.SequencerState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * The imageCoordinatorService is responsible for creating and managing
 * ImageCoordinators. In general an image coordinator is created when start
 * integration is received, and lasts until all of the corresponding FITS files
 * have been received, or a timeout occurs. It is considered the <i>current</i>
 * image coordinator between the startIntegration command and when the sequencer
 * ends after the endIntegation command.
 *
 * @author tonyj
 */
class ImageCoordinatorService implements HasLifecycle, ImageCoordinatorUtilities {

    private static final Logger LOG = Logger.getLogger(ImageCoordinatorService.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private FocalPlaneSubsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageDatabaseService idbs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private WebHooksConfig webHooksConfig;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentExecutionService executionService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageMessageHandling imageMessageHandling;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private SequencerConfig sequencerConfig;

    private final Map<String, ImageCoordinator> coordinators = new HashMap<>();
    private final Map<String, ScheduledFuture<?>> timeouts = new HashMap<>();
    private WebHooks webHooks;
    private ImageCoordinator currentImageCoordinator;
    private ImageCoordinator currentMetaDataTarget;
    private ImageCoordinator mostRecentImageCoordinator;
    private final Map<String, Serializable> savedMetaData = new HashMap();

    @Override
    public void start() {
        if (webHooksConfig != null) {
            webHooks = new WebHooks(executionService, webHooksConfig);
        }
    }

    @Override
    @SuppressWarnings("Convert2Lambda")
    public void postInit() {
        agentStateService.addStateChangeListener(new StateChangeListener() {
            
            private volatile ScheduledFuture<Void> scheduledIdleFlush;
            private volatile CCSTimeStamp lastClear;

            @Override
            public void stateChanged(CCSTimeStamp transitionTime, Object changedObj, Enum<?> newState, Enum<?> oldState) {
                LOG.log(Level.INFO, () -> String.format("FocalPlaneState change from %s to %s with timestamp %s", oldState, newState, transitionTime));
                
                if (newState == FocalPlaneState.QUIESCENT) {
                    long idleFlushTimeout = sequencerConfig.getIdleFlushTimeout();
                    if (idleFlushTimeout > 0) {
                        scheduledIdleFlush = subsys.getScheduler().schedule(() -> {
                            synchronized (agentStateService.getStateLock()) {
                                if (agentStateService.isInState(FocalPlaneState.QUIESCENT))  {
                                    subsys.getSequencers().startIdleFlush();
                                }
                            }
                            return null;
                        }, idleFlushTimeout, TimeUnit.MILLISECONDS);
                    }                   
                } else if (scheduledIdleFlush != null) {
                    scheduledIdleFlush.cancel(false);
                }
                
                if (oldState == FocalPlaneState.CLEARING || oldState == FocalPlaneState.READING_OUT) {
                    lastClear = transitionTime;
                } 
                
                if (newState == FocalPlaneState.READING_OUT) {
                    double darkTime = (transitionTime.getTAIDouble() - lastClear.getTAIDouble());
                    LOG.log(Level.INFO, "Setting darktime to {0}", darkTime);
                    subsys.setHeaderKeywords(Collections.singletonMap("darkTime", darkTime));
                    currentImageCoordinator.startReadout(darkTime, transitionTime, lastClear);
                    currentMetaDataTarget = null;
                } else if (newState == FocalPlaneState.INTEGRATING) {
                    currentImageCoordinator.startIntegrating(transitionTime);
                }
            }
        }, FocalPlaneState.class);
        agentStateService.addStateChangeListener(new StateChangeListener() {
            @Override
            public void stateChanged(CCSTimeStamp transitionTime, Object changedObj, Enum<?> newState, Enum<?> oldState) {
                LOG.log(Level.INFO, () -> String.format("SequencerState change from %s to %s with timestamp %s", oldState, newState, transitionTime));
            }
        } , SequencerState.class);

        try {
            sequencerConfig.getStore().addImageListener(new ImageListener() {
                @Override
                public void imageCreated(Image image) {
                    // Ignore images created in folders we don't care about.
                    if (image.getMetaData().getCreationFolderName().equals(sequencerConfig.getDAQFolder())) {
                        ImageCoordinator ic = coordinators.get(image.getMetaData().getName());
                        if (ic != null) {
                            ic.imageCreated(image);
                        } else {
                            LOG.log(Level.WARNING, "No image coordinator for image {0}", image.getMetaData().getName());
                        }
                    }
                }

                @Override
                public void imageComplete(Image image) {
                    // Ignore images created in folders we don't care about.
                    if (image.getMetaData().getCreationFolderName().equals(sequencerConfig.getDAQFolder())) {
                        ImageCoordinator ic = coordinators.get(image.getMetaData().getName());
                        if (ic != null) {
                            ic.imageComplete(image);
                        } else {
                            LOG.log(Level.WARNING, "No image coordinator for image {0}", image.getMetaData().getName());
                        }
                    }
                }
            });
        } catch (DAQException x) {
            throw new RuntimeException("Failed to initialize DAQ listener", x);
        }
    }

    /**
     * Called when a startIntegration event is received. Any meta-data which was
     * received since the last image is associated with the new image at this
     * time.
     *
     * @param in The new image name
     * @return The created ImageCoordinator, which also becomes the
     * currentImageCoordinator
     */
    ImageCoordinator createImageCoordinator(ImageName in) {
        ImageCoordinator coord = new ImageCoordinator(in, this, webHooks, idbs.getImageDatabase(), imageMessageHandling.getCount());
        if (!savedMetaData.isEmpty()) {
            coord.setMetaData(savedMetaData);
            savedMetaData.clear();
            currentMetaDataTarget = coord;
        }
        coordinators.put(in.toString(), coord);
        currentImageCoordinator = coord;
        mostRecentImageCoordinator = coord;
        return coord;
    }

    /**
     * Get the image coordinator associated with the given image name,
     *
     * @param imageName The image named to lookup
     * @return The associatedImageCoordinator
     * @throws RuntimeException if no image coordinator is currently associated
     * with this image.
     */
    ImageCoordinator getImageCoordinator(String imageName) {
        ImageCoordinator coord = coordinators.get(imageName);
        if (coord == null) {
            throw new RuntimeException("No iamge coordinator for " + imageName);
        }
        return coord;
    }

    /**
     * Gets the currentImageCoordinator, if it exists.
     *
     * @return The current image coordinator
     * @throws RuntimeException If there is no current image coordinator
     */
    ImageCoordinator getCurrentImageCoordinator() {
        if (currentImageCoordinator == null) {
            throw new RuntimeException("No image in progress");
        }
        return currentImageCoordinator;
    }

    /**
     * Gets the mostRecentImageCoordinator, if it exists.
     *
     * @return The current image coordinator
     * @throws RuntimeException If there is no current image coordinator
     */
    ImageCoordinator getMostRecentImageCoordinator() {
        if (mostRecentImageCoordinator == null) {
            throw new RuntimeException("No image in progress");
        }
        return mostRecentImageCoordinator;
    }

    /**
     * Called when the endIntegration command is received
     *
     * @param readout The imageReadout mode
     * @param image The image. This may be <code>null</code> if no image is
     * going to be created.
     */
    void endIntegration(LSE71Commands.ReadoutMode readout, Image image) {
        final ImageCoordinator imageCoordinator = currentImageCoordinator;
        final String imageName = imageCoordinator.getImageName().toString();
        // Set up a 60 seconds timeout
        timeouts.put(imageName, subsys.getScheduler().schedule(() -> {
            LOG.log(Level.WARNING, "Timed out waiting for image completion for {0}", imageCoordinator.getImageName());
            done(imageCoordinator);
        }, 60, TimeUnit.SECONDS));
        imageCoordinator.endIntegration(readout, image);
    }

    @Override
    public void sendEvent(String key, Serializable event) {
        subsys.sendEvent(key, event);
    }

    @Override
    public void setStateIf(FocalPlaneState currentState, FocalPlaneState newState) {
        subsys.setStateIf(currentState, newState);
    }

    // Called either by the image coordinator itself when it is finished, or as a result
    // of a timeout.
    @Override
    public void done(ImageCoordinator ic) {
        final String imageName = ic.getImageName().toString();
        LOG.log(Level.INFO, "Image coordinator for {0} complete", imageName);
        ScheduledFuture<?> futureTimeout = timeouts.get(imageName);
        if (futureTimeout != null) {
            futureTimeout.cancel(false);
        }
        if (currentImageCoordinator == ic) {
            currentImageCoordinator = null;
        }
        if (currentMetaDataTarget == ic) {
            currentMetaDataTarget = null;
        }
        // Avoid getting stuck im IMAGE_WAIT if DAQ did not report image complete (LSSTCCSRAFTS-595)
        // Care is needed because we could be in IMAGE_WAIT for a subsequent image if this image_coordinator
        // has already received imageComplete.
        if (!ic.isImageComplete()) {
            setStateIf(FocalPlaneState.IMAGE_WAIT, FocalPlaneState.QUIESCENT);
        }
        coordinators.remove(imageName);
    }

    void sequencersFinished() {
        synchronized (agentStateService.getStateLock()) {
            subsys.setState(currentImageCoordinator.sequencersFinished());
        }
        currentImageCoordinator = null;
    }

    /**
     * Add meta data to the next image. If this method is called between
     * startIntegration and endIntegration the metadata will be associated with
     * the current image, otherwise it will be associated with the next image..
     *
     * @param headersMap Set of meta-data to associate with the image.
     */
    void addMetaData(Map<String, Serializable> headersMap) {
        ImageCoordinator target = currentMetaDataTarget;
        if (target != null) {
            target.setMetaData(headersMap);
        } else {
            savedMetaData.putAll(headersMap);
        }
    }
}
