package org.lsst.ccs.subsystem.focalplane;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;

/**
 * Focal Plane level commands. These are normal level commands required by
 * LSE-71.
 *
 * @author The LSST CCS Team
 */
public class LSE71Commands {

    private final FocalPlaneSubsystem subsys;
    public enum READOUT_MODE { TRUE, FALSE, PSEUDO };


    LSE71Commands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    // Top level commands corresponding to LSE-71
    /**
     * Performs a clear operation. This method returns immediately and does not
     * wait for the sequencer to stop.
     *
     * @param nClears
     */
    @Command(type = Command.CommandType.ACTION, description = "Perform a clear operation", level = Command.NORMAL, autoAck = false)
    public void clear(int nClears) {
        subsys.helper()
                .precondition(nClears > 0 && nClears <= 15, "Invalid nClears: %d", nClears)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().clear(nClears);
                });
    }

    /**
     * Starts an integration operation. This method returns immediately.
     * @return The image name allocated
     */
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public ImageName startIntegration() {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .enterFaultOnException(true)
                .action(() -> {
                    ImageName in = subsys.getImageNameService().getImageName();
                    subsys.getSequencers().startIntegration(in.toString());
                    return in;
                });
    }
    
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public ImageName startIntegration(ImageName name) {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().startIntegration(name.toString());
                    return name;
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "Shift n rows", level = Command.NORMAL, autoAck = false)
    public void shiftNRows(int nRows) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().rowShift(nRows);
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "End exposure, and optionally read out", level = Command.NORMAL, autoAck = false)
    public void endIntegration(@Argument(defaultValue = "TRUE") READOUT_MODE readout) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getImageHandling().clearWait();
                    subsys.getSequencers().endIntegration(readout);
                });
    }
}
