package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Phaser;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.ReadoutMode;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.FitsFilesWrittenEvent;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * This class deals with receiving information about a particular image and
 * coordinating the sending of various events. It allows for the fact that some
 * information (e.g. FITSFiles) could be received after a startIntegration for a
 * following image.
 *
 * @author tonyj
 */
public class ImageCoordinator {

    private static final Logger LOG = Logger.getLogger(ImageCoordinator.class.getName());

    private final ImageName imageName;
    private CCSTimeStamp lastClear;
    private double darkTime;
    private CCSTimeStamp integrationEnd;
    private Image image;
    private final Phaser imageMetaDataEventCountdown;
    private CCSTimeStamp integrationStart;
    private final Phaser fitsFileCountdown;
    private final Phaser imageCountdown;
    private final int imageHandlersCount;
    private ImageDatabase.ImageDAO imageDAO;
    private Phaser imageDatabaseCountdown;
    private FileList fitsFiles = new FileList();
    private Exception fitsFileException;
    private final Phaser imageDoneCountdown;
    private final ImageCoordinatorUtilities utilities;

    /**
     * Constructor used to create image coordinator. Image coordinators are
     * normally created by the ImageCoordinatorService.
     *
     * @param name The image name
     * @param utilities Image utility object to be used for callbacks
     * @param webhooks Webhooks for notifying others (visualization) about new
     * images
     * @param imageDatabase The image database to be filled in with imagte
     * details
     * @param imageHandlersCount The number of image handlers we expect to
     * report FITS file generation (can be zero)
     */
    ImageCoordinator(ImageName name, ImageCoordinatorUtilities utilities, WebHooks webhooks, ImageDatabase imageDatabase, int imageHandlersCount) {
        this.imageName = name;
        this.imageHandlersCount = imageHandlersCount;
        this.utilities = utilities;

        // Used to track when all processing of this image is complete.
        imageDoneCountdown = new Phaser(4) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                utilities.done(ImageCoordinator.this);
                return true;
            }
        };

        if (imageDatabase != null) {
            imageDAO = imageDatabase.start(imageName);
            // ImageDatabaseCountdown is used to track when all required intomation has been collected to update
            // the iumage database
            imageDatabaseCountdown = new Phaser(5) {
                @Override
                protected boolean onAdvance(int phase, int registeredParties) {
                    imageDAO.commit();
                    imageDoneCountdown.arrive();
                    return true;
                }
            };
            imageDoneCountdown.register();
        }
        // ImageMetaDatazEventCountdown is use to track when all infoprmation required for generating the 
        // imageMetaDataEvent has been collected.
        imageMetaDataEventCountdown = new Phaser(3) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                ImageMetaDataEvent event = new ImageMetaDataEvent(imageName, darkTime, Long.toHexString(image.getMetaData().getId()), lastClear, integrationStart, integrationEnd);
                utilities.sendEvent(ImageMetaDataEvent.EVENT_KEY, event);
                imageDoneCountdown.arrive();
                return true;
            }
        };

        // fitsFileCountdown is used to track when all information about FITSFiles has been received.
        // If imageHandlersCount is zero, we need special handling
        fitsFileCountdown = new Phaser(this.imageHandlersCount == 0 ? 1 : this.imageHandlersCount) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                if (imageDAO != null) {
                    imageDAO.add(fitsFiles);
                    imageDatabaseCountdown.arrive();
                }
                if (webhooks != null) {
                    webhooks.notifyNewImage(imageName);
                }
                imageDoneCountdown.arrive();
                return true;
            }
        };
        // used to track when the DAQ has reported the image is un the 2-day store. 
        imageCountdown = new Phaser(1) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                utilities.setStateIf(FocalPlaneState.IMAGE_WAIT, FocalPlaneState.QUIESCENT);
                // If there are no image handlers we have as many FITS files as we will ever get (zero)
                if (imageHandlersCount == 0) {
                    fitsFileCountdown.arrive();
                }
                imageDoneCountdown.arrive();
                return true;
            }
        };
    }

    /**
     * Called from start integration command
     *
     * @param annotation The annotation provided with the command
     * @param locations The locations requested
     */
    void startIntegration(String annotation, LocationSet locations) {
        if (imageDAO != null) {
            imageDAO.setAnnotation(annotation);
            imageDAO.setLocations(locations);
            imageDatabaseCountdown.arrive();
        }
    }

    /**
     * Called as a result of the endIntegration command being executed.
     *
     * @param readoutMode The readout mode
     * @param image The image. (May be null if no image will be created)
     */
    void endIntegration(ReadoutMode readoutMode, Image image) {
        switch (readoutMode) {
            case FALSE:
            case PSEUDO:
                // We are done and do not expect to get any more notifications. 
                // We will not send any events or update the image database
                // We do not generate an imageMetaDataEvent
                // Calling waitForImage or waitForFITSFiles will hang and then timeout.
                utilities.done(this);
                break;
            default:
                // Normal case
                this.image = image;
                imageMetaDataEventCountdown.arrive();
                if (imageDAO != null) {
                    if (image != null) {
                        imageDAO.setDaqTag(image.getMetaData().getId());
                    }
                    imageDatabaseCountdown.arrive();
                }
        }
    }

    /**
     * Called when the image is created in the 2-day store. Not currently used.
     *
     * @param image The corresponding image
     */
    void imageCreated(Image image) {
    }

    /**
     * Called when the image is complete in the 2-day store.
     *
     * @param image The corresponding image
     */
    void imageComplete(Image image) {
        imageCountdown.arrive();
    }

    /**
     * Called when the image handlers notify us of new FITS files. Will be
     * called once for each image handler.
     *
     * @param fitsFileEvent
     */
    void addFitsFiles(FitsFilesWrittenEvent fitsFileEvent) {
        FileList fileList = fitsFileEvent.getFileList();
        //If fileList is null, it means there was an exception.
        if (fileList == null) {
            fitsFileException = fitsFileEvent.getException();
            // Terminating the fitsFileCountdown will immediately release any one who is waiting
            // in it, but means other actions that would have happened in FitsFileCountdown will not
            // occur.
            fitsFileCountdown.forceTermination();
        } else {
            fitsFiles.addAll(fitsFileEvent.getFileList());
            fitsFileCountdown.arrive();
        }
    }

    ImageName getImageName() {
        return imageName;
    }

    FileList waitForFITSFiles(Duration timeout) throws InterruptedException, TimeoutException, ExecutionException {
        fitsFileCountdown.awaitAdvanceInterruptibly(0, timeout.toMillis(), TimeUnit.MILLISECONDS);
        if (fitsFileException != null) {
            throw new ExecutionException("Error while waiting for FITS filee", fitsFileException);
        }
        return fitsFiles;
    }

    void waitForImages(Duration timeout) throws TimeoutException, InterruptedException {
        imageCountdown.awaitAdvanceInterruptibly(0, timeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    /**
     * Called when the sequencers finish after readout (but not after PSEUDO readout).
     * Depending on whether the
     * 2-day store has already reported that the image is complete, the state is
     * changed to either QUIESCENT or IMAGE_WAIT.
     *
     * @return The new state.
     */
    FocalPlaneState determineStateAfterReadout() {
        imageDoneCountdown.arrive();
        if (imageCountdown.isTerminated()) {
            return FocalPlaneState.QUIESCENT;
        } else {
            return FocalPlaneState.IMAGE_WAIT;
        }
    }

    /**
     * Called at the beginning of readout
     *
     * @param darkTime The computed darkTime
     * @param integrationEnd The timestamp for the end of integration.
     * @param lastClear The effective time for the last clear.
     */
    void startReadout(double darkTime, CCSTimeStamp integrationEnd, CCSTimeStamp lastClear) {
        this.darkTime = darkTime;
        this.integrationEnd = integrationEnd;
        this.lastClear = lastClear;
        imageMetaDataEventCountdown.arrive();
        if (imageDAO != null) {
            imageDAO.setDarkTime(darkTime);
            imageDatabaseCountdown.arrive();
        }
    }

    /**
     * Called at the beginning of integration.
     *
     * @param integrationStart The timestamp for the beginning of integration,
     */
    void startIntegrating(CCSTimeStamp integrationStart) {
        this.integrationStart = integrationStart;
        imageMetaDataEventCountdown.arrive();
        if (imageDAO != null) {
            imageDAO.setObsDate(integrationStart.getTAIInstant());
            imageDatabaseCountdown.arrive();
        }
    }

    /**
     * Optionally called to add meta-data for the image database. This meta-data
     * can either come from the MCM or from the script being used to run the
     * data taking.
     *
     * @param metaData The meta-data to be added to the image database.
     */
    void setMetaData(Map<String, Serializable> metaData) {
        LOG.log(Level.INFO, "Adding meta-data for image {0}: {1}", new Object[]{imageName, metaData});
        if (imageDAO != null) {
            imageDAO.addMetaData(metaData);
        }
    }

}
