package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.daq.ims.Camera;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.ImageMetaData;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.ReadoutMode;
import org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent;
import org.lsst.ccs.subsystem.focalplane.data.SequencerInfo;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Reb.RebType;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Encapsulates the behavior of all Sequencers in the focal plane.
 *
 * @author tonyj
 */
public class Sequencers {

    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());

    //TODO: Should these names be read from configuration?
    private static final String CLEAR_MAIN = "Clear";
    private static final String INTEGRATE_MAIN = "Integrate";
    private static final String ROW_SHIFT_R_MAIN = "RowShiftR";
    private static final String ROW_SHIFT_F_MAIN = "RowShiftF";
    private static final String READ_MAIN = "Read";
    private static final String PSEUDO_READ_MAIN = "PsuedoRead"; // Note mispelling used in sequencer
    private static final String[] MAINS = {CLEAR_MAIN, INTEGRATE_MAIN, ROW_SHIFT_R_MAIN, ROW_SHIFT_F_MAIN, READ_MAIN, PSEUDO_READ_MAIN};

    private static final String CLEAR_COUNT_PARAMETER = "ClearCount";
    private static final String SHIFT_COUNT_PARAMETER = "ShiftCount";
    private static final String[] PARAMETERS = {CLEAR_COUNT_PARAMETER, SHIFT_COUNT_PARAMETER};
    private final Lock sequencerLock = new ReentrantLock();
    private volatile Condition sequencerFinished;

    // Map from RebType to LocationType. We could get rid of this if RebType had a getLocationType() method
    private static final Map<RebType, Location.LocationType> REB_TYPE_MAP = new HashMap<>();

    static {
        REB_TYPE_MAP.put(RebType.SCIENCE, Location.LocationType.SCIENCE);
        REB_TYPE_MAP.put(RebType.WAVEFRONT, Location.LocationType.WAVEFRONT);
        REB_TYPE_MAP.put(RebType.GUIDER, Location.LocationType.GUIDER);
    }

    private final Map<Reb, SequencerProc> REBs = new LinkedHashMap<>();
    private final Map<Location, Reb> locationMap = new LinkedHashMap<>();
    private ImageName imageName;
    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig sequencerConfig;
    private Map<SequencerType, ModelAndName> models;
    private Map<String, Integer> opcodeMap;
    private Integer currentOpcode;
    private Boolean supportsMultiMains;
    private Map<Reb.RebType, int[]> registerMap;
    private KeyValueDataList sequencerKeyValueData;
    private String annotation;
    private LocationSet locations;
    private final LocationSet configuredLocations = new LocationSet();

    Sequencers(FocalPlaneSubsystem subsys, SequencerConfig sequencer) {
        this.subsys = subsys;
        this.sequencerConfig = sequencer;
    }

    void add(Reb reb, SequencerProc seq) {
        REBs.put(reb, seq);
        Location location = reb.getLocation();
        configuredLocations.add(location);
        locationMap.put(location, reb);
        if (subsys != null && subsys.dataProviderDictionaryService != null) {
            subsys.dataProviderDictionaryService.registerClass(SequencerInfo.class, reb.getFullName());
            subsys.dataProviderDictionaryService.registerClass(ImageReadoutParametersEvent.class, ImageReadoutParametersEvent.EVENT_KEY);
        }
    }

    public void clear(int nClears) throws REBException, RaftException, DAQException {
        setMain(CLEAR_MAIN);
        setParameter(CLEAR_COUNT_PARAMETER, nClears);
        sequencerFinished = scheduleEmulatedSequencer(Duration.ofMillis(nClears * 100));
        runSequencers(false, FocalPlaneState.CLEARING);
    }

    public void startIntegration(ImageName imageName, String annotation, LocationSet locations) throws REBException, RaftException, DAQException {
        this.imageName = imageName;
        this.annotation = annotation;
        this.locations = locations;
        setMain(INTEGRATE_MAIN);
        if (sequencerConfig.hasEmulatedDAQ()) {
            sequencerFinished = sequencerLock.newCondition();
        }
        runSequencers(false, FocalPlaneState.INTEGRATING, () -> {});
    }

    public void rowShift(int nRows) throws REBException, RaftException, DAQException {
        if (nRows == 0) {
            return;
        } else if (nRows > 0) {
            setMain(ROW_SHIFT_F_MAIN);
        } else {
            setMain(ROW_SHIFT_R_MAIN);
        }
        setParameter(SHIFT_COUNT_PARAMETER, Math.abs(nRows));
        sequencerFinished = null;
        runSequencers(false, FocalPlaneState.ROW_SHIFT);
    }

    public void stop() throws DAQException, REBException {
        if (sequencerConfig.hasEmulatedDAQ()) {
            Condition condition = sequencerFinished;
            if (condition != null) {
                sequencerLock.lock();
                try {
                    condition.signalAll();
                } finally {
                    sequencerLock.unlock();
                }
            }
        } else if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            sequencerConfig.getCamera().startSequencer(Camera.OPCODE_STOP);
        } else {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().sendStop();
                }
            }
        }
    }

    public Image endIntegration(ReadoutMode readout) throws REBException, RaftException, DAQException {
        stop();
        waitForStop(Duration.ofSeconds(1));
        switch (readout) {
            case TRUE:
                setMain(READ_MAIN);
                sequencerFinished = scheduleEmulatedSequencer(Duration.ofSeconds(2));
                return runSequencers(true, FocalPlaneState.READING_OUT, () -> {
                    if (subsys != null) {
                        subsys.getImageCoordinatorService().determineStateAfterReadout();
                    }
                });
            case PSEUDO:
                setMain(PSEUDO_READ_MAIN);
                sequencerFinished = scheduleEmulatedSequencer(Duration.ofSeconds(2));
                runSequencers(false, FocalPlaneState.READING_OUT);
                return null;
            default:
            case FALSE:
                sequencerFinished = null;
                setState(FocalPlaneState.QUIESCENT);
                // Nothing else to do
                return null;
        }
    }

    private Condition scheduleEmulatedSequencer(Duration when) {
        if (sequencerConfig.hasEmulatedDAQ()) {
            Condition condition = sequencerLock.newCondition();
            doLater(when, () -> {
                sequencerLock.lock();
                try {
                    condition.signalAll();
                } finally {
                    sequencerLock.unlock();
                }
            });
            return condition;
        } else {
            return null;
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        currentOpcode = opcodeMap.get(mainName);
        if (currentOpcode == null) {
            throw new RaftException("Invalid main: " + mainName);
        }
        if ((!Objects.equals(Boolean.TRUE, supportsMultiMains)) || subsys == null) {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().setStart(mainName);
                }
            }
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            if (isRebOnlineAndValid(entry.getKey())) {
                entry.getValue().setParameter(parameter, value);
            }
        }
    }

    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, stateWhileRunning, FocalPlaneState.QUIESCENT);
    }
    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning, FocalPlaneState stateOnCompletion) throws REBException, RaftException, DAQException {
        return runSequencers(acquire, stateWhileRunning, () -> setState(stateOnCompletion));
    } 
    
    private Image runSequencers(boolean acquire, FocalPlaneState stateWhileRunning, Runnable runOnFinish) throws REBException, RaftException, DAQException {
        LocationSet effectiveLocations = locations == null || locations.isEmpty() ? configuredLocations : locations;
        Image image = null;
        if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            if (currentOpcode == null) {
                throw new REBException("opcode not set");
            }
            LOG.log(Level.INFO, "Starting sequencers with opcode={0}", currentOpcode);
            if (acquire) {
                ImageMetaData imd = new ImageMetaData(imageName.toString(), sequencerConfig.getDAQFolder(), annotation, currentOpcode, effectiveLocations);
                image = sequencerConfig.getCamera().triggerImage(imd);
                sendImageReadoutParametersEvent(effectiveLocations);
            } else {
                sequencerConfig.getCamera().startSequencer(currentOpcode);
            }
        } else if (subsys != null) {
            int oldOpcode = 1; // Opcode to be used with pre-multi-main firmware 
            if (acquire) {
                ImageMetaData imd = new ImageMetaData(imageName.toString(), sequencerConfig.getDAQFolder(), annotation, currentOpcode, effectiveLocations);
                image = sequencerConfig.getCamera().triggerImage(imd);
                sendImageReadoutParametersEvent(effectiveLocations);
            } else {
                sequencerConfig.getCamera().startSequencer(oldOpcode);
            }
        } else {
            // Used for simulation only
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().startSequencer();
                }
            }
            if (acquire) {
                sendImageReadoutParametersEvent(effectiveLocations);
            }
        }
        //TODO: What happens if above fails, state should not be left unchanged.
        setState(stateWhileRunning);
        doInBackground(() -> {
            try {
                // No way to tell how long we have to wait for infinite loop, 
                // so wait a somewhat arbitrary 1,000,000 seconds.
                waitForStop(Duration.ofMillis(1_000_000_000));
                LOG.info("Sequencers finished");
                runOnFinish.run();
            } catch (REBException ex) {
                LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
            }
        });
        return image;
    }

    private void sendImageReadoutParametersEvent(LocationSet effectiveLocations) throws RaftException, REBException {
        ImageReadoutParametersEvent irpe
                = new ImageReadoutParametersEvent(imageName, annotation, effectiveLocations, currentOpcode, sequencerConfig.getDAQFolder());
        for (Location location : effectiveLocations) {
            Reb reb = locationMap.get(location);
            SequencerProc seq = REBs.get(reb);
            int preCols = seq.getParameter("PreCols");
            int underCols = seq.getParameter("UnderCols");
            int readCols = seq.getParameter("ReadCols");
            int postCols = seq.getParameter("PostCols");
            int readCols2 = seq.getParameter("ReadCols2");
            int overCols = seq.getParameter("OverCols");
            int preRows = seq.getParameter("PreRows");
            int readRows = seq.getParameter("ReadRows");
            int postRows = seq.getParameter("PostRows");
            int overRows = seq.getParameter("OverRows");
            irpe.setParameters(location, reb.getCCDType(), preCols, underCols, readCols, postCols, readCols2, overCols,
                    preRows, readRows, postRows, overRows);
        }
        if (subsys != null) {
            subsys.sendEvent(ImageReadoutParametersEvent.EVENT_KEY, irpe);
        }
    }

    public void waitForStop(Duration timeout) throws REBException {
        if (sequencerConfig.hasEmulatedDAQ()) {
            try {
                Condition condition = sequencerFinished;
                if (condition != null) {
                    sequencerLock.lock();
                    try {
                        condition.await((int) timeout.toMillis(), TimeUnit.MILLISECONDS);
                    } finally {
                        sequencerLock.unlock();
                    }
                }
            } catch (InterruptedException ex) {
                throw new REBException("Interrupt while waiting for sequencer");
            }
        } else {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    SequencerProc seq = entry.getValue();
                    // NOTE: waitDone throws a REBException it if times out
                    seq.waitDone((int) timeout.toMillis());
                    int errorAddr = seq.getErrorAddr();
                    if (errorAddr != -1) {
                        LOG.log(Level.SEVERE, "REB error register set to {0} for {1}", new Object[]{errorAddr, entry.getKey()});
                        //TODO: We should go into fault state here
                    }
                }
            }
        }
    }

    /**
     * Fail fast if the sequencers loaded do not meet our minimal requirements
     *
     * @param seq The sequencer to test
     * @throws RaftException If any of the required mains or parameters are
     * missing
     */
    private static void sanityCheck(FPGA2Model model) throws RaftException, REBException {
        Map<String, Integer> mainMap = model.getMainAddresses();
        for (String main : MAINS) {
            if (mainMap.get(main) == null) {
                throw new RaftException("Sequencer failed sanity check, missing main " + main);
            }
        }
        List<FPGA2Model.PointerInfo> pointers = model.getPointers();
        for (String parameter : PARAMETERS) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (pointer.getName().equals(parameter)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
            }
        }
    }

    /**
     * Check that all required meta-data parameters are present in the
     * sequencer.
     *
     * @param metaData A list of required parameters
     * @param model The sequencer model to check
     * @return The list of addresses of the meta-data registers
     * @throws RaftException If one or more meta-data parameters are missing or
     * invalid
     */
    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map<String, PointerInfo> pointerMap = model.getPointerMap();
        int k = 0;
        for (String metaName : metaDataNames) {
            PointerInfo pi = pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION:
                        registers[k++] = pi.getAddress();
                        break;
                    case REPEAT_SUBROUTINE:
                        registers[k++] = pi.getAddress();
                        break;
                    default:
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());

                }
            } else {
                throw new RaftException("Required parameter not defined " + metaName);
            }
        }
        LOG.log(Level.INFO, "Meta-data registers set {0}", Arrays.toString(registers));
        return registers;
    }

    static FPGA2Model validate(String sequencerFile) throws RaftException, REBException, IOException {
        InputStream input = BootstrapResourceUtils.getBootstrapResource(sequencerFile);
        if (input == null) {
            throw new IOException("Invalid sequencer file name " + sequencerFile);
        }
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        FPGA2Model compiledModel = builder.compileFile(input);

        sanityCheck(compiledModel);
        return compiledModel;
    }

    /**
     * Called to load the configured sequencers into all registered
     * SequencerProcs.
     *
     * @throws RaftException
     * @throws REBException
     */
    void load() throws RaftException, REBException, DAQException {
        Map<SequencerType, ModelAndName> localModels = new LinkedHashMap<>();
        Map<Reb.RebType, int[]> localRegisterMap = new LinkedHashMap<>();
        opcodeMap = null;

        Map<SequencerType, String> sequencers = sequencerConfig.getSequencers();
        for (Map.Entry<SequencerType, String> sequencer : sequencers.entrySet()) {
            try {
                FPGA2Model model = validate(sequencer.getValue());
                localModels.put(sequencer.getKey(), new ModelAndName(model, sequencer.getValue()));
            } catch (IOException x) {
                RaftException re = new RaftException("Error reading sequencer " + sequencer.getValue());
                re.initCause(x);
                throw re;
            }
        }
        SequencerInfo data;
        this.sequencerKeyValueData = new KeyValueDataList();

        // Now loop over the REBS, and load the appropriate sequencer for each one
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            Reb reb = entry.getKey();
            SequencerProc seq = entry.getValue();
            ModelAndName modelAndName = getModelAndNameForREB(localModels, reb);
            int[] registers = computeMetaDataRegisters(sequencerConfig.getMetaDataRegisters(), modelAndName.getModel());
            if (localRegisterMap.get(reb.getRebType()) == null) {
                localRegisterMap.put(reb.getRebType(), registers);
            } else if (!Arrays.equals(localRegisterMap.get(reb.getRebType()), registers)) {
                throw new RaftException("Sequencers have inconsistent meta-data register sets for type " + reb.getRebType());
            }
            if (opcodeMap == null) {
                opcodeMap = modelAndName.getModel().getMainOpcodes();
            } else if (!opcodeMap.equals(modelAndName.getModel().getMainOpcodes())) {
                throw new RaftException("Sequencers have inconsistent main opcodes\n" + opcodeMap + "\n" + modelAndName.getModel().getMainOpcodes());
            }
            if (isRebOnlineAndValid(reb)) {
                try {
                    FirmwareVersion version = getFirmwareVersionForReb(reb);
                    checkAndLoadSequencer(seq, reb, modelAndName, version);
                } catch (REBException | RaftException x) {
                    // TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error stopping or loading sequencer " + modelAndName.getName() + " into " + reb.getFullName(), x);
                }
            }
            // This data will go to the status aggregator in the image-handlers
            data = new SequencerInfo(modelAndName.getName(), String.valueOf(modelAndName.getModel().computeCheckSum()));
            this.sequencerKeyValueData.addData(reb.getFullName(), data);
        }

        this.models = localModels;
        this.registerMap = localRegisterMap;

        if (subsys != null) {
            for (Map.Entry<Reb.RebType, int[]> registersMapEntry : localRegisterMap.entrySet()) {
                sequencerConfig.getCamera().setRegisterList(REB_TYPE_MAP.get(registersMapEntry.getKey()), registersMapEntry.getValue());
            }
            subsys.publishSubsystemDataOnStatusBus(sequencerKeyValueData);
        }
    }

    private void checkAndLoadSequencer(SequencerProc seq, Reb reb, ModelAndName modelAndName, FirmwareVersion version) throws REBException, RaftException {
        // First check to see if sequencer is running, and if so attempt to stop it
        if (seq.isRunning()) {
            LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
            seq.sendStop();
            // NOTE: waitDone throws a REBException it if times out
            seq.waitDone(2000); // 2 seconds timeout
        }
        boolean rebSupportsMultiMains = version.supportsMultiMains();
        if (supportsMultiMains == null) {
            supportsMultiMains = rebSupportsMultiMains;
        } else if (supportsMultiMains != rebSupportsMultiMains) {
            LOG.log(Level.SEVERE, "Inconsistent firmware version {0} for REB {1}", new Object[]{version, reb.getFullName()});
        }
        LOG.log(Level.INFO, "Loading sequencer {0} into {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
        seq.loadSequencer(modelAndName.getModel());
        loadSequencerParameters(seq, modelAndName.getModel());
    }

    private void doInBackground(Runnable run) {
        Thread t = new Thread(run);
        t.setName("Sequencer wait thread");
        t.start();
    }

    private void doLater(Duration when, Runnable runnable) {
        doInBackground(() -> {
            try {
                Thread.sleep(when.toMillis());
                runnable.run();
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt", ex);
            }
        });
    }

    private void setState(FocalPlaneState focalPlaneState) {
        if (subsys != null) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(focalPlaneState);
        }
    }

    Map<SequencerType, ModelAndName> getModels() {
        return models;
    }

    Map<RebType, int[]> getRegisterMap() {
        return registerMap;
    }

    KeyValueDataList getSequencerKeyValueData() {
        return sequencerKeyValueData;
    }

    /**
     * Load the compiled sequencer into the specified SequencerProc. This is
     * called when a REB comes online after the initial load of sequencers
     *
     * @param sequencer The SequencerProc to load.
     */
    void load(Reb reb, FirmwareVersion version) {
        if (models != null) {
            SequencerProc seq = REBs.get(reb);
            ModelAndName modelAndName = getModelAndNameForREB(models, reb);
            try {
                checkAndLoadSequencer(seq, reb, modelAndName, version);
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Unable to load sequencer into " + reb.getFullName(), ex);
            }
        }
    }

    private boolean isRebOnlineAndValid(Reb reb) {
        if (subsys == null) {
            return true;
        } else {
            return subsys.isRebOnlineAndValid(reb);
        }
    }

    private FirmwareVersion getFirmwareVersionForReb(Reb reb) {
        if (subsys == null) {
            return new FirmwareVersion(0x31385004);
        } else {
            return subsys.getFirmwareVersionForReb(reb);
        }
    }

    private static ModelAndName getModelAndNameForREB(Map<SequencerType, ModelAndName> models, Reb reb) {
        String rebTypeString = reb.getRebType() + "-" + reb.getCCDType().getName().toUpperCase();
        switch (rebTypeString) {
            case "SCIENCE-E2V":
                ModelAndName result = models.get(SequencerType.SCIENCE_E2V);
                if (result == null) {
                    result = models.get(SequencerType.E2V);
                }
                return result;
            case "SCIENCE-ITL":
                result = models.get(SequencerType.SCIENCE_ITL);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "WAVEFRONT-WF":
                result = models.get(SequencerType.WAVEFRONT);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "GUIDER-ITL":
                result = models.get(SequencerType.GUIDER);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            default:
                throw new IllegalArgumentException("Unknown sequencer type: " + rebTypeString);
        }
    }

    void loadSequencerParameters() {
        REBs.forEach((reb, seq) -> {
            ModelAndName modelAndNameForREB = getModelAndNameForREB(models, reb);
            try {
                loadSequencerParameters(seq, modelAndNameForREB.getModel());
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Failed to set sequencer parameter for " + reb.getFullName(), ex);
            }
        });
    }

    void loadSequencerParameters(SequencerProc seq, FPGA2Model model) throws REBException, RaftException {
        sequencerConfig.loadParameters(model, seq);
    }

    LocationSet getConfiguredLocations() {
        return configuredLocations;
    }

    private static class ModelAndName {

        private final FPGA2Model model;
        private final String name;

        public ModelAndName(FPGA2Model model, String name) {
            this.model = model;
            this.name = name;
        }

        public FPGA2Model getModel() {
            return model;
        }

        public String getName() {
            return name;
        }
    }
}
