package org.lsst.ccs.subsystem.focalplane;

import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.imagehandling.data.FitsFilesWrittenEvent;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;

/**
 * Deals with receiving images from ImageHandling subsystem.
 * @author tonyj
 */
class ImageMessageHandling implements AgentPresenceListener, StatusMessageListener, HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageMessageHandling.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private FocalPlaneSubsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageCoordinatorService ics;

    private final Set<AgentInfo> imageHandlers = new CopyOnWriteArraySet<>();

    @Override
    public void start() {
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        subsys.getMessagingAccess().addStatusMessageListener(this, BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
    }

    @Override
    public void shutdown() {
        subsys.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        subsys.getMessagingAccess().removeStatusMessageListener(this);
    }

    private static boolean isAgentAnImageHandler(AgentInfo agentInfo) {
        return AgentCategory.IMAGE_HANDLER.name().equals(agentInfo.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY));
    }

    @Override
    public void connected(AgentInfo... agents) {
        boolean imageHandlerConnected = false;
        for (AgentInfo a : agents) {
            if (isAgentAnImageHandler(a)) {
                imageHandlers.add(a);
                imageHandlerConnected = true;
            }
        }
        // When a new image handler joins, we need to republish the sequencer info (LSSTCCSRAFTS-465)
        if (imageHandlerConnected) {
            KeyValueDataList sequencerKeyValueData = subsys.getSequencers().getSequencerKeyValueData();
            if (sequencerKeyValueData != null) {
                subsys.publishSubsystemDataOnStatusBus(sequencerKeyValueData);
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo a : agents) {
            if (isAgentAnImageHandler(a)) {
                imageHandlers.remove(a);
            }
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        AgentInfo agentInfo = msg.getOriginAgentInfo();
        if (isAgentAnImageHandler(agentInfo)) {
            StatusSubsystemData statusData = (StatusSubsystemData) msg;
            String key = statusData.getDataKey();
            switch (key) {
                case ImageReceivedEvent.EVENT_KEY:
                    ImageReceivedEvent imageEvent = (ImageReceivedEvent) statusData.getSubsystemData().getValue();
                    LOG.log(Level.INFO, "Received " + ImageReceivedEvent.EVENT_KEY + " for {0} from {1}", new Object[]{imageEvent.getImageName(), agentInfo.getName()});
                    break;
                case FitsFilesWrittenEvent.EVENT_KEY:
                    FitsFilesWrittenEvent fitsFileEvent = (FitsFilesWrittenEvent) statusData.getSubsystemData().getValue();
                    LOG.log(Level.INFO, "Received " + FitsFilesWrittenEvent.EVENT_KEY + " for {0} from {1} with {2} files", new Object[]{fitsFileEvent.getImageName(), agentInfo.getName(), fitsFileEvent.getFileList().size()});
                    ImageCoordinator coord = ics.getImageCoordinator(fitsFileEvent.getImageName());
                    coord.addFitsFiles(fitsFileEvent);
                    break;
            }
        }
    }

    int getCount() {
        return imageHandlers.size();
    }
}
