package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.StateChangeListener;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.ims.DAQException;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.daq.ims.ImageListener;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.services.AgentExecutionService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * The imageCoordinatorService is responsible for creating and managing
 * ImageCoordinators
 *
 * @author tonyj
 */
class ImageCoordinatorService implements HasLifecycle, ImageCoordinatorUtilities {

    private static final Logger LOG = Logger.getLogger(ImageCoordinatorService.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private FocalPlaneSubsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageDatabaseService idbs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private WebHooksConfig webHooksConfig;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentExecutionService executionService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageMessageHandling imageMessageHandling;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private SequencerConfig sequencerConfig;

    private final Map<String, ImageCoordinator> coordinators = new HashMap<>();
    private WebHooks webHooks;
    private ImageCoordinator currentImageCoordinator;
    private ImageCoordinator currentMetaDataTarget;
    private CCSTimeStamp lastClear;
    private final Map<String, Serializable> savedMetaData = new HashMap();

    @Override
    public void start() {
        if (webHooksConfig != null) {
            webHooks = new WebHooks(executionService, webHooksConfig);
        }
    }

    @Override
    public void postInit() {
        agentStateService.addStateChangeListener(new StateChangeListener() {
            @Override
            public void stateChanged(CCSTimeStamp transitionTime, Object changedObj, Enum<?> newState, Enum<?> oldState) {
                if (newState == FocalPlaneState.QUIESCENT
                        && (oldState == FocalPlaneState.CLEARING || oldState == FocalPlaneState.READING_OUT)) {
                    lastClear = transitionTime;
                } else if (newState == FocalPlaneState.READING_OUT) {
                    double darkTime = (transitionTime.getTAIDouble() - lastClear.getTAIDouble());
                    LOG.log(Level.INFO, "Setting darktime to {0}", darkTime);
                    subsys.setHeaderKeywords(Collections.singletonMap("darkTime", darkTime));
                    currentImageCoordinator.startReadout(darkTime, transitionTime, lastClear);
                    currentMetaDataTarget = null;
                } else if (newState == FocalPlaneState.INTEGRATING) {
                    currentImageCoordinator.startIntegrating(transitionTime);
                }
            }

        }, FocalPlaneState.class);

        try {
            sequencerConfig.getStore().addImageListener(new ImageListener() {
                @Override
                public void imageCreated(Image image) {
                    ImageCoordinator ic = coordinators.get(image.getMetaData().getName());
                    if (ic != null) {
                        ic.imageCreated(image);
                    } else {
                        LOG.log(Level.WARNING, "No image coordinator for image {0}", image.getMetaData().getName());
                    }
                }

                @Override
                public void imageComplete(Image image) {
                    ImageCoordinator ic = coordinators.get(image.getMetaData().getName());
                    if (ic != null) {
                        ic.imageComplete(image);
                    } else {
                        LOG.log(Level.WARNING, "No image coordinator for image {0}", image.getMetaData().getName());
                    }
                }
            });
        } catch (DAQException x) {
            throw new RuntimeException("Failed to initialize DAQ listener", x);
        }
    }

    ImageCoordinator createImageCoordinator(ImageName in) {
        ImageCoordinator coord = new ImageCoordinator(in, this, webHooks, idbs.getImageDatabase(), imageMessageHandling.getCount());
        if (!savedMetaData.isEmpty()) {
            coord.setMetaData(savedMetaData);
            savedMetaData.clear();
            currentMetaDataTarget = coord;
        }
        coordinators.put(in.toString(), coord);
        currentImageCoordinator = coord;
        return coord;
    }

    ImageCoordinator getImageCoordinator(String imageName) {
        ImageCoordinator coord = coordinators.get(imageName);
        if (coord == null) {
            throw new RuntimeException("No iamge coordinator for " + imageName);
        }
        return coord;
    }

    ImageCoordinator getCurrentImageCoordinator() {
        if (currentImageCoordinator == null) {
            throw new RuntimeException("No image in progress");
        }
        return currentImageCoordinator;
    }

    void endIntegration(LSE71Commands.ReadoutMode readout, Image image) {
        currentImageCoordinator.endIntegration(readout, image);
        if (readout == LSE71Commands.ReadoutMode.FALSE) {
            coordinators.remove(currentImageCoordinator.getImageName().toString());
        } else {
            // TODO: Set up a timer by which the image should be complete.
        }
    }

    @Override
    public void sendEvent(String key, Serializable event) {
        subsys.sendEvent(key, event);
    }

    @Override
    public void setStateIf(FocalPlaneState currentState, FocalPlaneState newState) {
        subsys.setStateIf(currentState, newState);
    }

    void determineStateAfterReadout() {
        subsys.setState(currentImageCoordinator.determineStateAfterReadout());
    }

    void addMetaData(Map<String, Serializable> headersMap) {
        ImageCoordinator target = currentMetaDataTarget;
        if (target != null) {
            target.setMetaData(headersMap);
        } else {
            savedMetaData.putAll(headersMap);
        }
    }
}
