package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Phaser;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import jline.internal.Log;
import org.lsst.ccs.daq.ims.Image;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.ReadoutMode;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.FitsFilesWrittenEvent;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * This class deals with receiving information about a particular image and
 * coordinating the sending of various events. It allows for the fact that some
 * information (e.g. FITSFiles) could be received after a startIntegration for a
 * following image.
 *
 * @author tonyj
 */
public class ImageCoordinator {

    private static final Logger LOG = Logger.getLogger(ImageCoordinator.class.getName());

    private final ImageName imageName;
    private CCSTimeStamp lastClear;
    private double darkTime;
    private CCSTimeStamp integrationEnd;
    private Image image;
    private final ImageCoordinatorUtilities utilities;
    private final Phaser imageMetaDataEventCountdown;
    private CCSTimeStamp integrationStart;
    private final Phaser fitsFileCountdown;
    private final Phaser imageCountdown;
    private final ImageDatabase imageDatabase;
    private final int imageHandlersCount;
    private ImageDatabase.ImageDAO imageDAO;
    private Phaser imageDatabaseCountdown;
    private FileList fitsFiles = new FileList();
    private Exception fitsFileException;

    ImageCoordinator(ImageName name, ImageCoordinatorUtilities utilities, WebHooks webhooks, ImageDatabase imageDatabase, int imageHandlersCount) {
        this.imageName = name;
        this.utilities = utilities;
        this.imageDatabase = imageDatabase;
        this.imageHandlersCount = imageHandlersCount;

        if (imageDatabase != null) {
            imageDAO = imageDatabase.start(imageName);
            imageDatabaseCountdown = new Phaser(5) {
                @Override
                protected boolean onAdvance(int phase, int registeredParties) {
                    imageDAO.commit();
                    return true;
                }
            };
        }

        imageMetaDataEventCountdown = new Phaser(3) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                ImageMetaDataEvent event = new ImageMetaDataEvent(imageName, darkTime, Long.toHexString(image.getMetaData().getId()), lastClear, integrationStart, integrationEnd);
                utilities.sendEvent(ImageMetaDataEvent.EVENT_KEY, event);
                return true;
            }
        };

        // If imageHandlersCount is zero, we need special handling
        fitsFileCountdown = new Phaser(this.imageHandlersCount == 0 ? 1 : this.imageHandlersCount) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                if (imageDAO != null) {
                    imageDAO.add(fitsFiles);
                    imageDatabaseCountdown.arrive();
                }
                if (webhooks != null) {
                    webhooks.notifyNewImage(imageName);
                }
                return true;
            }
        };

        imageCountdown = new Phaser(1) {
            @Override
            protected boolean onAdvance(int phase, int registeredParties) {
                utilities.setStateIf(FocalPlaneState.IMAGE_WAIT, FocalPlaneState.QUIESCENT);
                if (imageHandlersCount == 0) fitsFileCountdown.arrive();
                return true;
            }
        };
    }

    void startIntegration(String annotation, LocationSet locations) {
        if (imageDAO != null) {
            imageDAO.setAnnotation(annotation);
            imageDAO.setLocations(locations);
            imageDatabaseCountdown.arrive();
        }
    }

    void endIntegration(ReadoutMode readoutMode, Image image) {
        this.image = image;
        imageMetaDataEventCountdown.arrive();
        if (imageDAO != null) {
            imageDAO.setDaqTag(image.getMetaData().getId());
            imageDatabaseCountdown.arrive();
        }
    }

    void imageCreated(Image image) {
        LOG.log(Level.INFO, "Image created {0}", image);
    }

    void imageComplete(Image image) {
        LOG.log(Level.INFO, "Image complete {0}", image);
        imageCountdown.arrive();
    }

    void addFitsFiles(FitsFilesWrittenEvent fitsFileEvent) {
        FileList fileList = fitsFileEvent.getFileList();
        //If fileList is null, it means there was an exception.
        if (fileList == null) {
            fitsFileException = fitsFileEvent.getException();
            fitsFileCountdown.forceTermination();
        } else {
            fitsFiles.addAll(fitsFileEvent.getFileList());
            fitsFileCountdown.arrive();
        }
    }

    ImageName getImageName() {
        return imageName;
    }

    FileList waitForFITSFiles(Duration timeout) throws InterruptedException, TimeoutException, ExecutionException {
        fitsFileCountdown.awaitAdvanceInterruptibly(0, timeout.toMillis(), TimeUnit.MILLISECONDS);
        if (fitsFileException != null) {
            throw new ExecutionException("Error while waiting for FITS filee", fitsFileException);
        }
        return fitsFiles;
    }

    void waitForImages(Duration timeout) throws TimeoutException, InterruptedException {
        imageCountdown.awaitAdvanceInterruptibly(0, timeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    FocalPlaneState determineStateAfterReadout() {
        if (imageCountdown.isTerminated()) {
            return FocalPlaneState.QUIESCENT;
        } else {
            return FocalPlaneState.IMAGE_WAIT;
        }
    }

    void startReadout(double darkTime, CCSTimeStamp integrationEnd, CCSTimeStamp lastClear) {
        this.darkTime = darkTime;
        this.integrationEnd = integrationEnd;
        this.lastClear = lastClear;
        imageMetaDataEventCountdown.arrive();
        if (imageDAO != null) {
            imageDAO.setDarkTime(darkTime);
            imageDatabaseCountdown.arrive();
        }
    }

    void startIntegrating(CCSTimeStamp integrationStart) {
        this.integrationStart = integrationStart;
        imageMetaDataEventCountdown.arrive();
        if (imageDAO != null) {
            imageDAO.setObsDate(integrationStart.getTAIInstant());
            imageDatabaseCountdown.arrive();
        }
    }

    void setMetaData(Map<String, Serializable> metaData) {
        LOG.log(Level.INFO, "Adding meta-data for image {0}: {1}", new Object[]{imageName, metaData});
        if (imageDAO != null) {
            imageDAO.addMetaData(metaData);
        }
    }

}
