package org.lsst.ccs.subsystem.focalplane;

import java.util.Set;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Focal Plane level commands. These are normal level commands required by
 * LSE-71.
 *
 * @author The LSST CCS Team
 */
public class LSE71Commands {

    private static final LocationSet ALL_LOCATIONS = new LocationSet();

    private final FocalPlaneSubsystem subsys;

    public enum READOUT_MODE {
        TRUE, FALSE, PSEUDO
    };

    private ImageName currentImage;

    LSE71Commands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    // Top level commands corresponding to LSE-71
    /**
     * Performs a clear operation. This method returns immediately and does not
     * wait for the sequencer to stop.
     *
     * @param nClears
     */
    @Command(type = Command.CommandType.ACTION, description = "Perform a clear operation", level = Command.NORMAL, autoAck = false)
    public void clear(@Argument(defaultValue = "1") int nClears) {
        subsys.helper()
                .precondition(nClears > 0 && nClears <= 15, "Invalid nClears: %d", nClears)
                .precondition(FocalPlaneState.QUIESCENT, FocalPlaneState.NEEDS_CLEAR)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().clear(nClears);
                });
    }

    /**
     * Starts an integration operation. This method returns immediately.
     *
     * @param annotation The image annotation
     * @param locations The set of locations to read
     * @return The image name allocated
     */
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public ImageName startIntegration(
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        return subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .precondition(subsys.getImageNameService() != null, "Image name service not available")
                .enterFaultOnException(true)
                .action(() -> {
                    currentImage = subsys.getImageNameService().getImageName();
                    LocationSet localLocations = computeLocations(locations);
                    subsys.getSequencers().startIntegration(currentImage, annotation, localLocations);
                    return currentImage;
                });
    }

    /**
     * Starts an integration operation for a named image. This method returns
     * immediately.
     *
     * @param imageName The image name to use
     * @param annotation The image annotation
     * @param locations The set of locations to read
     */
    @Command(type = Command.CommandType.ACTION, description = "Start integration", level = Command.NORMAL, autoAck = false)
    public void startNamedIntegration(
            @Argument ImageName imageName,
            @Argument(defaultValue = "") String annotation,
            @Argument(defaultValue = Argument.NULL) Set locations) {
        subsys.helper()
                .precondition(FocalPlaneState.QUIESCENT)
                .enterFaultOnException(true)
                .action(() -> {
                    currentImage = imageName;
                    LocationSet localLocations = computeLocations(locations);
                    subsys.getSequencers().startIntegration(currentImage, annotation, localLocations);
                });

    }

    @Command(type = Command.CommandType.ACTION, description = "Shift n rows", level = Command.NORMAL, autoAck = false)
    public void shiftNRows(int nRows) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.getSequencers().rowShift(nRows);
                });
    }

    @Command(type = Command.CommandType.ACTION, description = "End exposure, and optionally read out", level = Command.NORMAL, autoAck = false)
    public void endIntegration(@Argument(defaultValue = "TRUE") READOUT_MODE readout) {
        subsys.helper()
                .precondition(FocalPlaneState.INTEGRATING)
                .enterFaultOnException(true)
                .action(() -> {
                    subsys.endIntegration(currentImage, readout);
                    subsys.getSequencers().endIntegration(readout);
                });
    }

    private LocationSet computeLocations(Set locations) {
        if (locations == null || locations.isEmpty()) {
            return ALL_LOCATIONS; // Note, REBDriver interprets empty set to mean "all"
        } else if (locations instanceof LocationSet) {
            return (LocationSet) locations;
        } else {
            LocationSet result = new LocationSet();
            for (Object o : locations) {
                if (o instanceof Location) {
                    result.add((Location) o);
                } else if (o != null) {
                    // Note, we use LocationSet.of to support Strings like R22, as opposed to 
                    // individial locations such as R22/Reb1
                    result.addAll(LocationSet.of(o.toString()));
                }
            }
            return result;
        }
    }
}
