package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.ConfigurationBulkChangeHandler;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.utilities.readout.ReadOutParametersNew;

/**
 *
 * @author tonyj
 */
@SuppressWarnings("FieldMayBeFinal")
public class SequencerConfig implements ConfigurationBulkChangeHandler {

    private static final int UNDEFINED = -1;

    @ConfigurationParameter(category = "Sequencer")
    private Map<SequencerType, String> sequencer = new HashMap<>();

    @ConfigurationParameter(category = "Sequencer", description = "Names of meta-data related registers which must be in the sequencer")
    private String[] metaDataRegisters = ReadOutParametersNew.DEFAULT_NAMES;

    @ConfigurationParameter(category = "Sequencer")
    private int preCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int underCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int readCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int postCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int readCols2 = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int overCols = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int preRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int readRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int postRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer")
    private int overRows = UNDEFINED;

    @ConfigurationParameter(category = "Sequencer", description = "True if the sequencer is in scan mode", units = "unitless")
    private boolean scanMode = false;

    @ConfigurationParameter(category = "DAQ", description = "Folder in 2-day store where images will be created", units = "unitless")
    private String daqFolder = "raw";

    @ConfigurationParameter(category = "DAQ", description = "True when using an emulated DAQ", units = "unitless")
    private boolean emulatedDAQ = false;

    public SequencerConfig() {
        sequencer.put(SequencerType.ITL, "FP_ITL_2s_ir2_v1.seq");
        sequencer.put(SequencerType.E2V, "FP_E2V_2s_ir2_v1.seq");
    }

    public Map<SequencerType, String> getSequencers() {
        return sequencer;
    }

    public String[] getMetaDataRegisters() {
        return metaDataRegisters;
    }

    // For test purposes only
    void setSequencer(String sequencer) {
        this.sequencer.put(SequencerType.ITL, sequencer);
    }

    public String getDAQFolder() {
        return daqFolder;
    }

    @Override
    public void validateBulkChange(Map<String, Object> parametersView) {
        Map<SequencerType, String> sequencerMap = (Map<SequencerType, String>) parametersView.get("sequencer");
        if (sequencerMap.isEmpty()) {
            throw new IllegalArgumentException("Invalid empty sequencer map");
        }
        for (String sequencerFile : sequencerMap.values()) {
            try {
                Sequencers.validate(sequencerFile);
            } catch (IOException | REBException | RaftException ex) {
                throw new IllegalArgumentException("Invalid sequencer file " + sequencerFile, ex);
            }
        }
    }

    void loadParameters(FPGA2Model model, SequencerProc seq) throws REBException, RaftException {
        setParameter(seq, model, "PreCols", preCols);
        setParameter(seq, model, "UnderCols", underCols);
        setParameter(seq, model, "ReadCols", readCols);
        setParameter(seq, model, "PostCols", postCols);
        setParameter(seq, model, "ReadCols2", readCols2);
        setParameter(seq, model, "OverCols", overCols);
        setParameter(seq, model, "PreRows", preRows);
        setParameter(seq, model, "ReadRows", readRows);
        setParameter(seq, model, "PostRows", postRows);
        setParameter(seq, model, "OverRows", overRows);
        seq.enableScan(scanMode);
    }

    private void setParameter(SequencerProc seq, FPGA2Model model, String name, int value) throws REBException, RaftException {
        PointerInfo pi = model.getPointerMap().get(name);
        if (pi == null) {
            throw new RaftException("Invalid pointer "+name);
        }
        seq.setParameter(name, value != UNDEFINED ? value : pi.getValue());
    }

    boolean hasEmulatedDAQ() {
        return emulatedDAQ;
    }
}
