package org.lsst.ccs.subsystem.focalplane;

import java.io.Serializable;
import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.imagehandling.data.FileList;
import org.lsst.ccs.subsystem.imagehandling.data.FitsFilesWrittenEvent;
import org.lsst.ccs.subsystem.imagehandling.data.ImageReceivedEvent;

/**
 *
 * @author tonyj
 */
class ImageMessageHandling implements AgentPresenceListener, StatusMessageListener, HasLifecycle {

    private static final Logger LOG = Logger.getLogger(ImageMessageHandling.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private FocalPlaneSubsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private ImageDatabaseService idbs;
    
    private final Set<AgentInfo> imageHandlers = new CopyOnWriteArraySet<>();    
    private String currentImageName;
    
    private AbortableCountDownLatch fitsFileCountDown;
    private AbortableCountDownLatch imageCountDown;
    private final FileList fitsFiles = new FileList();
    
    private final RuntimeException integrationStartException = new RuntimeException("Starting new acquisition");
    private ImageDatabase imageDatabase;

    @Override
    public void start() {    
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        subsys.getMessagingAccess().addStatusMessageListener(this, BusMessageFilterFactory.messageClass(StatusSubsystemData.class));
        imageDatabase = idbs.getImageDatabase();
    }

    @Override
    public void shutdown() {
        subsys.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        subsys.getMessagingAccess().removeStatusMessageListener(this);
    }

    void clearWait(ImageName currentImage) {
        currentImageName = currentImage.toString();
        LOG.log(Level.INFO, "Resetting counters used to wait on images and fits files for image {0}.", currentImageName);
        resetVariablesForCurrentIntegration();
        // Start database ingest
        if (imageDatabase != null) {
            imageDatabase.ingest(currentImage,fitsFiles,fitsFileCountDown);
        }
    }
    
    private void resetVariablesForCurrentIntegration() {
        // Abort previous countdown, in case someone is stil waiting
        if (fitsFileCountDown != null) {
            fitsFileCountDown.abort(integrationStartException);
        }
        if (imageCountDown != null) {
            imageCountDown.abort(integrationStartException);        
        }

        int countdownSize = imageHandlers.size();
        fitsFileCountDown = new AbortableCountDownLatch(countdownSize);
        imageCountDown = new AbortableCountDownLatch(countdownSize);

        fitsFiles.clear();
    }
    
    
    FileList waitForFITSFiles(Duration timeout) throws InterruptedException, TimeoutException {
        if (fitsFileCountDown.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
            return fitsFiles;
        } else {
            throw new TimeoutException("Timed out while waiting for FITS files");
        }
    }

    void waitForImages(Duration timeout) throws TimeoutException, InterruptedException {
        if (!imageCountDown.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
            throw new TimeoutException("Timed out while waiting for images");
        }
    }
    
    
    private static boolean isAgentAnImageHandler(AgentInfo agentInfo) {
        return AgentCategory.IMAGE_HANDLER.name().equals(agentInfo.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY));
    }
    
    @Override
    public void connected(AgentInfo... agents) {
        boolean imageHandlerConnected = false;
        for (AgentInfo a : agents) {
            if (isAgentAnImageHandler(a)) {
                imageHandlers.add(a);
                imageHandlerConnected = true;
            }
        }
        // When a new image handler joins, we need to republish the sequencer info (LSSTCCSRAFTS-465)
        if (imageHandlerConnected) {
            KeyValueDataList sequencerKeyValueData = subsys.getSequencers().getSequencerKeyValueData();
            if (sequencerKeyValueData != null) {
               subsys.publishSubsystemDataOnStatusBus(sequencerKeyValueData);
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo a : agents) {
            if (isAgentAnImageHandler(a)) {
                imageHandlers.remove(a);
            }
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        AgentInfo agentInfo = msg.getOriginAgentInfo();
        if ( isAgentAnImageHandler(agentInfo) ) {
            StatusSubsystemData statusData = (StatusSubsystemData)msg;
            String key = statusData.getDataKey();
            switch(key) {
                case ImageReceivedEvent.EVENT_KEY:
                    ImageReceivedEvent imageEvent = (ImageReceivedEvent) statusData.getSubsystemData().getValue();
                    //Make sure this event is related to the current image
                    if ( imageEvent.getImageName().equals(currentImageName)) {
                        if (imageDatabase != null) {
                            imageDatabase.setMetaData(imageEvent.getImageMetaData());
                        }
                        imageCountDown.countDown();
                        LOG.log(Level.INFO, "Received "+ImageReceivedEvent.EVENT_KEY+" for {0} from {1}", new Object[]{currentImageName, agentInfo.getName()});
                    }
                    break;
                case FitsFilesWrittenEvent.EVENT_KEY:
                    FitsFilesWrittenEvent fitsFileEvent = (FitsFilesWrittenEvent) statusData.getSubsystemData().getValue();
                    //Make sure this event is related to the current image
                    if ( fitsFileEvent.getImageName().equals(currentImageName)) {
                        FileList fileList = fitsFileEvent.getFileList();
                        //If fileList is null, it means there was an exception.
                        if ( fileList == null ) {
                            fitsFileCountDown.abort(fitsFileEvent.getException());
                        } else {
                            if (imageDatabase != null) {
                                imageDatabase.addLocations(fitsFileEvent.getLocations());
                            }
                            fitsFiles.addAll(fitsFileEvent.getFileList());                            
                            fitsFileCountDown.countDown();
                            LOG.log(Level.INFO, "Received "+FitsFilesWrittenEvent.EVENT_KEY+" for {0} from {1} with {2} files", new Object[]{currentImageName, agentInfo.getName(), fitsFileEvent.getFileList().size()});
                        }
                    }
                    break;
            }
        }
    }

    void handleMetaData(Map<String, Serializable> headersMap) {
        if (imageDatabase != null) {
            imageDatabase.addMetaData(headersMap);
        }
    }
}
