package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.KeyValueDataList;
import static org.lsst.ccs.drivers.reb.BaseSet.REB_TYPE_GUIDER;
import static org.lsst.ccs.drivers.reb.BaseSet.REB_TYPE_SCIENCE;
import static org.lsst.ccs.drivers.reb.BaseSet.REB_TYPE_WAVEFRONT;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.READOUT_MODE;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Reb.RebType;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * Encapsulates the behavior of all Sequencers in the focal plane.
 *
 * @author tonyj
 */
public class Sequencers {

    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());

    //TODO: Should these names be read from configuration?
    private static final String CLEAR_MAIN = "Clear";
    private static final String INTEGRATE_MAIN = "Integrate";
    private static final String ROW_SHIFT_R_MAIN = "RowShiftR";
    private static final String ROW_SHIFT_F_MAIN = "RowShiftF";
    private static final String READ_MAIN = "Read";
    private static final String PSEUDO_READ_MAIN = "PsuedoRead"; // Note mispelling used in sequencer
    private static final String[] MAINS = {CLEAR_MAIN, INTEGRATE_MAIN, ROW_SHIFT_R_MAIN, ROW_SHIFT_F_MAIN, READ_MAIN, PSEUDO_READ_MAIN};

    private static final String CLEAR_COUNT_PARAMETER = "ClearCount";
    private static final String SHIFT_COUNT_PARAMETER = "ShiftCount";
    private static final String[] PARAMETERS = {CLEAR_COUNT_PARAMETER, SHIFT_COUNT_PARAMETER};

    // Map from RebType to REB_TYPE used by BaseSet
    private static final Map<RebType, Integer> REB_TYPE_MAP = new HashMap<>();

    static {
        REB_TYPE_MAP.put(RebType.SCIENCE, REB_TYPE_SCIENCE);
        REB_TYPE_MAP.put(RebType.WAVEFRONT, REB_TYPE_WAVEFRONT);
        REB_TYPE_MAP.put(RebType.GUIDER, REB_TYPE_GUIDER);
    }

    private final Map<Reb, SequencerProc> REBs = new LinkedHashMap<>();
    private String imageName;
    private final FocalPlaneSubsystem subsys;
    private final SequencerConfig sequencerConfig;
    private Map<SequencerType, ModelAndName> models;
    private Map<String, Integer> opcodeMap;
    private Integer currentOpcode;
    private Boolean supportsMultiMains;
    private Map<Reb.RebType, int[]> registerMap;
    private KeyValueDataList sequencerKeyValueData;
    private String annotation;
    private LocationSet locations;

    Sequencers(FocalPlaneSubsystem subsys, SequencerConfig sequencer) {
        this.subsys = subsys;
        this.sequencerConfig = sequencer;
    }

    void add(Reb reb, SequencerProc seq) {
        REBs.put(reb, seq);
    }

    public void clear(int nClears) throws REBException, RaftException {
        setMain(CLEAR_MAIN);
        setParameter(CLEAR_COUNT_PARAMETER, nClears);
        run(false, FocalPlaneState.CLEARING);
    }

    public void startIntegration(String imageName, String annotation, LocationSet locations) throws REBException, RaftException {
        this.imageName = imageName;
        this.annotation = annotation;
        this.locations = locations;
        setMain(INTEGRATE_MAIN);
        run(false, FocalPlaneState.INTEGRATING);
    }

    public void rowShift(int nRows) throws REBException, RaftException {
        if (nRows == 0) {
            return;
        } else if (nRows > 0) {
            setMain(ROW_SHIFT_F_MAIN);
        } else {
            setMain(ROW_SHIFT_R_MAIN);
        }
        setParameter(SHIFT_COUNT_PARAMETER, Math.abs(nRows));
        run(false, FocalPlaneState.ROW_SHIFT);
    }

    public void stop() throws REBException {
        if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            subsys.getGlobalProc().startSequencer(31);
        } else {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().sendStop();
                }
            }
        }
    }

    public void endIntegration(READOUT_MODE readout) throws REBException, RaftException {
        stop();
        waitForStop(Duration.ofSeconds(1));
        switch (readout) {
            case TRUE:
                setMain(READ_MAIN);
                run(true, FocalPlaneState.READING_OUT);
                break;
            case PSEUDO:
                setMain(PSEUDO_READ_MAIN);
                run(false, FocalPlaneState.READING_OUT);
                break;
            case FALSE:
            // Nothing to do
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        currentOpcode = opcodeMap.get(mainName);
        if (currentOpcode == null) {
            throw new RaftException("Invalid main: " + mainName);
        }
        if ((!Objects.equals(Boolean.TRUE, supportsMultiMains)) || subsys == null) {
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().setStart(mainName);
                }
            }
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            if (isRebOnlineAndValid(entry.getKey())) {
                entry.getValue().setParameter(parameter, value);
            }
        }
    }

    private void run(boolean acquire, FocalPlaneState stateOnSuccess) throws REBException {
        if ((Objects.equals(Boolean.TRUE, supportsMultiMains)) && subsys != null) {
            if (currentOpcode == null) {
                throw new REBException("opcode not set");
            }
            LOG.log(Level.INFO, "Starting sequencers with opcode={0}", currentOpcode);
            if (acquire) {
                subsys.getGlobalProc().acquireImage(imageName, sequencerConfig.getDAQFolder(),
                        currentOpcode, annotation, locations == null ? new int[0] : locations.asIntArray());
            } else {
                subsys.getGlobalProc().startSequencer(currentOpcode);
            }
        } else if (subsys != null) {
            int oldOpcode = 1; // Opcode to be used with pre-multi-main firmware 
            if (acquire) {
                subsys.getGlobalProc().acquireImage(imageName, sequencerConfig.getDAQFolder(),
                        oldOpcode, annotation, locations == null ? new int[0] : locations.asIntArray());
            } else {
                subsys.getGlobalProc().startSequencer(oldOpcode);
            }
        } else {
            // Used for simulation only
            for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
                if (isRebOnlineAndValid(entry.getKey())) {
                    entry.getValue().startSequencer();
                }
            }

        }
        //TODO: What happens if above fails, state should not be left unchanged.
        setState(stateOnSuccess);
        doInBackground(() -> {
            try {
                // No way to tell how long we have to wait for infinite loop, 
                // so wait a somewhat arbitrary 1,000,000 seconds.
                waitForStop(Duration.ofMillis(1_000_000_000));
                setState(FocalPlaneState.QUIESCENT);
                LOG.info("Sequencers finished");
            } catch (REBException ex) {
                LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
            }
        });
    }

    public void waitForStop(Duration timeout) throws REBException {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            if (isRebOnlineAndValid(entry.getKey())) {
                SequencerProc seq = entry.getValue();
                // NOTE: waitDone throws a REBException it if times out
                seq.waitDone((int) timeout.toMillis());
                int errorAddr = seq.getErrorAddr();
                if (errorAddr != -1) {
                    LOG.log(Level.SEVERE, "REB error register set to {0} for {1}", new Object[]{errorAddr, entry.getKey()});
                    //TODO: We should go into fault state here
                }
            }
        }
    }

    /**
     * Fail fast if the sequencers loaded do not meet our minimal requirements
     *
     * @param seq The sequencer to test
     * @throws RaftException If any of the required mains or parameters are
     * missing
     */
    private static void sanityCheck(FPGA2Model model) throws RaftException, REBException {
        Map<String, Integer> mainMap = model.getMainAddresses();
        for (String main : MAINS) {
            if (mainMap.get(main) == null) {
                throw new RaftException("Sequencer failed sanity check, missing main " + main);
            }
        }
        List<FPGA2Model.PointerInfo> pointers = model.getPointers();
        for (String parameter : PARAMETERS) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (pointer.getName().equals(parameter)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
            }
        }
    }

    /**
     * Check that all required meta-data parameters are present in the
     * sequencer.
     *
     * @param metaData A list of required parameters
     * @param model The sequencer model to check
     * @return The list of addresses of the meta-data registers
     * @throws RaftException If one or more meta-data parameters are missing or
     * invalid
     */
    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map<String, PointerInfo> pointerMap = model.getPointerMap();
        int k = 0;
        for (String metaName : metaDataNames) {
            PointerInfo pi = pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION:
                        registers[k++] = pi.getAddress();
                        break;
                    case REPEAT_SUBROUTINE:
                        registers[k++] = pi.getAddress();
                        break;
                    default:
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());

                }
            } else {
                throw new RaftException("Required parameter not defined " + metaName);
            }
        }
        LOG.log(Level.INFO, "Meta-data registers set {0}", Arrays.toString(registers));
        return registers;
    }

    static FPGA2Model validate(String sequencerFile) throws RaftException, REBException, IOException {
        InputStream input = BootstrapResourceUtils.getBootstrapResource(sequencerFile);
        if (input == null) {
            throw new IOException("Invalid sequencer file name " + sequencerFile);
        }
        FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
        FPGA2Model compiledModel = builder.compileFile(input);

        sanityCheck(compiledModel);
        return compiledModel;
    }

    /**
     * Called to load the configured sequencers into all registered
     * SequencerProcs.
     *
     * @throws RaftException
     * @throws REBException
     */
    void load() throws RaftException, REBException {
        Map<SequencerType, ModelAndName> localModels = new LinkedHashMap<>();
        Map<Reb.RebType, int[]> localRegisterMap = new LinkedHashMap<>();
        opcodeMap = null;

        Map<SequencerType, String> sequencers = sequencerConfig.getSequencers();
        for (Map.Entry<SequencerType, String> sequencer : sequencers.entrySet()) {
            try {
                FPGA2Model model = validate(sequencer.getValue());
                localModels.put(sequencer.getKey(), new ModelAndName(model, sequencer.getValue()));
            } catch (IOException x) {
                RaftException re = new RaftException("Error reading sequencer " + sequencer.getValue());
                re.initCause(x);
                throw re;
            }
        }
        KeyValueDataList data = new KeyValueDataList();
        // Now loop over the REBS, and load the appropriate sequencer for each one
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            Reb reb = entry.getKey();
            SequencerProc seq = entry.getValue();
            ModelAndName modelAndName = getModelAndNameForREB(localModels, reb);
            int[] registers = computeMetaDataRegisters(sequencerConfig.getMetaDataRegisters(), modelAndName.getModel());
            if (localRegisterMap.get(reb.getRebType()) == null) {
                localRegisterMap.put(reb.getRebType(), registers);
            } else if (!Arrays.equals(localRegisterMap.get(reb.getRebType()), registers)) {
                throw new RaftException("Sequencers have inconsistent meta-data register sets for type " + reb.getRebType());
            }
            if (opcodeMap == null) {
                opcodeMap = modelAndName.getModel().getMainOpcodes();
            } else if (!opcodeMap.equals(modelAndName.getModel().getMainOpcodes())) {
                throw new RaftException("Sequencers have inconsistent main opcodes\n" + opcodeMap + "\n" + modelAndName.getModel().getMainOpcodes());
            }
            if (isRebOnlineAndValid(reb)) {
                try {
                    // First check to see if sequencer is running, and if so attempt to stop it
                    if (seq.isRunning()) {
                        LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
                        seq.sendStop();
                        // NOTE: waitDone throws a REBException it if times out
                        seq.waitDone(2000); // 2 seconds timeout
                    }
                    LOG.log(Level.INFO, "Loading sequencer {0} into {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
                    seq.loadSequencer(modelAndName.getModel());
                } catch (REBException x) {
                    // TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error stopping or loading sequencer " + modelAndName.getName() + " into " + reb.getFullName(), x);
                }
            }
            // This data will go to the status aggregator in the image-handlers
            data.addData(reb.getFullName() + "/sequencerFile", modelAndName.getName());
            data.addData(reb.getFullName() + "/sequencerChecksum", String.valueOf(modelAndName.getModel().computeCheckSum()));
        }
        this.models = localModels;
        this.registerMap = localRegisterMap;
        this.sequencerKeyValueData = data;

        if (subsys != null) {
            for (Map.Entry<Reb.RebType, int[]> registersMapEntry : localRegisterMap.entrySet()) {
                subsys.getGlobalProc().setRegisters(REB_TYPE_MAP.get(registersMapEntry.getKey()), registersMapEntry.getValue());
            }
            subsys.publishSubsystemDataOnStatusBus(data);
        }
    }

    private void doInBackground(Runnable run) {
        Thread t = new Thread(run);
        t.setName("Sequencer wait thread");
        t.start();
    }

    private void setState(FocalPlaneState focalPlaneState) {
        if (subsys != null) {
            subsys.getAgentService(AgentStateService.class).updateAgentState(focalPlaneState);
        }
    }

    Map<SequencerType, ModelAndName> getModels() {
        return models;
    }

    Map<RebType, int[]> getRegisterMap() {
        return registerMap;
    }

    KeyValueDataList getSequencerKeyValueData() {
        return sequencerKeyValueData;
    }

    /**
     * Load the compiled sequencer into the specified SequencerProc. This is
     * called when a REB comes online after the initial load of sequencers
     *
     * @param sequencer The SequencerProc to load.
     */
    void load(Reb reb, FirmwareVersion version) {
        if (models != null) {
            SequencerProc seq = REBs.get(reb);
            ModelAndName modelAndName = getModelAndNameForREB(models, reb);
            try {
                // First check to see if sequencer is running, and if so attempt to stop it
                if (seq.isRunning()) {
                    LOG.log(Level.INFO, "Sequencer {0} running, attempting to stop", reb.getFullName());
                    seq.sendStop();
                    // NOTE: waitDone throws a REBException it if times out
                    seq.waitDone(2000); // 2 seconds timeout
                }
                boolean rebSupportsMultiMains = version.supportsMultiMains();
                if (supportsMultiMains == null) {
                    supportsMultiMains = rebSupportsMultiMains;
                } else if (supportsMultiMains != rebSupportsMultiMains) {
                    LOG.log(Level.SEVERE, "Inconsistent firmware version {0} for REB {1}", new Object[]{version, reb.getFullName()});
                }
                seq.loadSequencer(modelAndName.getModel());
                loadSequencerParameters(seq, modelAndName.getModel());
                LOG.log(Level.INFO, "Loading sequencer {0} into REB {1}", new Object[]{modelAndName.getName(), reb.getFullName()});
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Unable to load sequencer into " + reb.getFullName(), ex);
            }
        }
    }

    private boolean isRebOnlineAndValid(Reb reb) {
        if (subsys == null) {
            return true;
        } else {
            return subsys.isRebOnlineAndValid(reb);
        }
    }

    private static ModelAndName getModelAndNameForREB(Map<SequencerType, ModelAndName> models, Reb reb) {
        String rebTypeString = reb.getRebType() + "-" + reb.getCCDType().getName().toUpperCase();
        switch (rebTypeString) {
            case "SCIENCE-E2V":
                ModelAndName result = models.get(SequencerType.SCIENCE_E2V);
                if (result == null) {
                    result = models.get(SequencerType.E2V);
                }
                return result;
            case "SCIENCE-ITL":
                result = models.get(SequencerType.SCIENCE_ITL);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "WAVEFRONT-WF":
                result = models.get(SequencerType.WAVEFRONT);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            case "GUIDER-ITL":
                result = models.get(SequencerType.GUIDER);
                if (result == null) {
                    result = models.get(SequencerType.ITL);
                }
                return result;
            default:
                throw new IllegalArgumentException("Unknown sequencer type: " + rebTypeString);
        }
    }

    void loadSequencerParameters() {
        for (Map.Entry<Reb, SequencerProc> entry : REBs.entrySet()) {
            Reb reb = entry.getKey();
            ModelAndName modelAndNameForREB = getModelAndNameForREB(models, reb);
            SequencerProc seq = entry.getValue();
            try {
                loadSequencerParameters(seq, modelAndNameForREB.getModel());
            } catch (REBException | RaftException ex) {
                //TODO: Enter fault state?
                LOG.log(Level.SEVERE, "Failed to set sequencer parameter for " + reb.getFullName(), ex);
            }
        }
    }

    void loadSequencerParameters(SequencerProc seq, FPGA2Model model) throws REBException, RaftException {
        sequencerConfig.loadParameters(model, seq);
    }

    private static class ModelAndName {

        private final FPGA2Model model;
        private final String name;

        public ModelAndName(FPGA2Model model, String name) {
            this.model = model;
            this.name = name;
        }

        public FPGA2Model getModel() {
            return model;
        }

        public String getName() {
            return name;
        }
    }
}
