package org.lsst.ccs.subsystem.focalplane;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.actions.RebPowerAction;
import org.lsst.ccs.subsystem.common.focalplane.data.HasFocalPlaneData;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.GlobalProc;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;

public class FocalPlaneSubsystem extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevices = new LinkedHashMap<>();

    private Sequencers sequencers;
    private final FocalPlane focalPlaneGeometry;
    private ImageMessageHandling imageMessageHandling;
    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystem.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalProc globalProc;

    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dictionaryService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private SequencerConfig sequencerConfig;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageMessageHandling ImageMessageHandling;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService agentAlertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Agent agent;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentCommandDictionaryService agentCommandDictionaryService;

    private ComponentLookup lookupService;
    private long lastClear;

    public FocalPlaneSubsystem(FocalPlane geometry) {
        super("fp", AgentInfo.AgentType.WORKER);
        this.focalPlaneGeometry = geometry;
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

    @Override
    public void build() {
        agentPropertiesService.setAgentProperty(HasFocalPlaneData.AGENT_PROPERTY, HasFocalPlaneData.generatePropertyValue(FocalPlaneDataGroup.class));
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.FOCAL_PLANE.name());
        agentPropertiesService.setAgentProperty(RebPowerAction.class.getCanonicalName(), "uses");

        lookupService = agent.getComponentLookup();
        ComponentNode thisComponent = lookupService.getComponentNodeForObject(this);

        imageMessageHandling = new ImageMessageHandling();
        ComponentNode imageMessageHandlingNode = new ComponentNode("imageMessageHandling", imageMessageHandling);
        lookupService.addComponentNodeToLookup(thisComponent, imageMessageHandlingNode);

        for (Raft r : focalPlaneGeometry.getChildrenList()) {

            ComponentNode raftNode = new ComponentNode(r.getName(), new HardwareIdConfiguration());
            lookupService.addComponentNodeToLookup(thisComponent, raftNode);            
            for (Reb reb : r.getRebs()) {
                String rebName = r.getName() + "/" + reb.getName();

                ComponentNode rebHardwareNode = new ComponentNode(reb.getName()+"_hardware", new HardwareIdConfiguration());
                lookupService.addComponentNodeToLookup(raftNode, rebHardwareNode);            

                ComponentNode rebComponent = lookupService.getNodeByPath(rebName);
                for (CCD ccd : reb.getCCDs()) {
                    String sensorName = ccd.getName();
                    ComponentNode sensorNode = new ComponentNode(sensorName, new CCDHardwareIdConfiguration());
                    lookupService.addComponentNodeToLookup(rebComponent, sensorNode);
                }
            }
        }
        //Register the FocalPlaneState
        agentStateService.registerState(FocalPlaneState.class, "The state of the Focal Plane", this);
    }

    @Override
    public void postInit() {

        sequencers = new Sequencers(this, sequencerConfig);
        agentStateService.updateAgentState(FocalPlaneState.NEEDS_CLEAR);

        rebDevices.forEach((rebName, device) -> {
            Reb reb = focalPlaneGeometry.getReb(rebName);
            device.setRebGeometry(reb);
            sequencers.add(reb, device.getSequencer());
        });

        agentStateService.addStateChangeListener(
                (Object device, Enum<?> newState, Enum<?> oldState) -> {
                    if (newState == RebValidationState.VALID) {
                        final REBDevice rebDevice = (REBDevice) device;
                        String path = lookupService.getNameOfComponent(device);
                        try {
                            FirmwareVersion version = new FirmwareVersion(rebDevice.getHwVersion());
                                LOG.log(Level.INFO, "Disabled LAM for firmware {0}", version);
                                // FIXME: Temporary workaround for LSSTCCSRAFTS-529
                                try {
                                    rebDevice.setRegister(0x17, new int[]{0});
                                } catch (RaftException x) {
                                    LOG.log(Level.SEVERE, "Could not disable LAM for "+rebDevice.getName(), x);
                            }

                            sequencers.load(focalPlaneGeometry.getReb(path), version);

                            Object result = rebDevice.loadAspics(true);
                            LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{rebDevice.getName(), result});

                        } catch (Exception x) {
                            LOG.log(Level.WARNING, "Exception while loading ASPICS", x);
                        }
                    }
                },
                 RebDeviceState.class,
                 RebValidationState.class
        );

        agentCommandDictionaryService.addCommandSetToObject(
                new LSE71Commands(this), "");
        agentCommandDictionaryService.addCommandSetToObject(
                new ScriptingCommands(this), "");
        agentCommandDictionaryService.addCommandSetToObject(
                new FocalPlaneCommands(this), "");
        final FocalPlaneMetaDataCommands metaDataCommands = new FocalPlaneMetaDataCommands(this, ImageMessageHandling);

        agentCommandDictionaryService.addCommandSetToObject(metaDataCommands,
                "");

        agentStateService.addStateChangeListener(
                (Object changedObj, Enum<?> newState, Enum<?> oldState) -> {
                    if (newState == FocalPlaneState.QUIESCENT
                    && (oldState == FocalPlaneState.CLEARING || oldState == FocalPlaneState.READING_OUT)) {
                        // TODO: We should probably be using CCSTimesstamp here
                        lastClear = System.currentTimeMillis();
                    } else if (newState == FocalPlaneState.READING_OUT) {
                        double darkTime = (System.currentTimeMillis() - lastClear) / 1000.;
                        LOG.log(Level.INFO, "Setting darktime to {0}", darkTime);
                        metaDataCommands.setHeaderKeyword("DarkTime", darkTime);
                    }
                },
                 FocalPlaneState.class);
        
        //This is needed to publish the Alert summary when a Power Subsystem 
        //joins the buses.
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(
                new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo ai : agents) {
                    if (ai.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.POWER.name())) {
                        LOG.log(Level.INFO, "Publishing raised alert summary for {0}", new Object[]{ai.getName()});
                        getScheduler().execute(() -> {
                            agentAlertService.publishRaisedAlertSummary();
                        });
                        break;
                    }
                }
            }

        }
        );
    }

    @Override
    public void preStart() {
        //Manipulate the dictionary here: 
        for (DataProviderInfo data : dictionaryService.getDataProviderDictionary().getDataProviderInfos()) {
            FocalPlaneDataGroup dataGroup = FocalPlaneDataGroup.findFocalPlaneDataGroup(data);
            if (dataGroup != null) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }

    @Override
    public void postStart() {
        configService.addConfigurationListener(new ConfigListener(this));
        loadSequencers();
    }

    void loadSequencers() {
        try {
            sequencers.load();
        } catch (RaftException | REBException x) {
            //TODO: Go into fault state?
            throw new RuntimeException("Error configuring sequencers", x);
        }
    }

    void loadASPICS() {
        for (REBDevice device : rebDevices.values()) {
            if (device.isOnline() && device.isSerialNumValid()) {
                try {
                    Object result = device.loadAspics(true);
                    LOG.log(Level.INFO, "Loaded aspics for {0} result {1}", new Object[]{device.getFullName(), result});
                } catch (Exception ex) {
                    //TODO: Go into fault state?
                    LOG.log(Level.SEVERE, "Error while loading ASPICS", ex);
                }
            }
        }
    }

    boolean isRebOnlineAndValid(Reb reb) {
        return stateService.isComponentInState(reb.getFullName(), RebValidationState.VALID);
    }

    public Sequencers getSequencers() {
        return sequencers;
    }

    GlobalProc getGlobalProc() {
        return globalProc;
    }

    ImageNameService getImageNameService() {
        return imageNameService;
    }

    ImageMessageHandling getImageMessageHandling() {
        return imageMessageHandling;
    }

    Map<String, REBDevice> getRebDevices() {
        return rebDevices;
    }

    void loadSequencerParameters() {
        sequencers.loadSequencerParameters();
    }

}
