package org.lsst.ccs.subsystem.focalplane;

import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import static org.lsst.ccs.drivers.reb.BaseSet.REB_TYPE_SCIENCE;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.subsystem.focalplane.LSE71Commands.READOUT_MODE;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.SequencerData;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2Model.PointerInfo;
import org.lsst.ccs.subsystem.rafts.fpga.compiler.FPGA2ModelBuilder;

/**
 * Encapsulates the behavior of all Sequencers in the focal plane. Currently
 * ignores difference between science rafts and wavefront sensors. Currently
 * ignores guiders.
 *
 * @author tonyj
 */
public class Sequencers {

    private static final Logger LOG = Logger.getLogger(Sequencers.class.getName());

    //TODO: Should these names be read from configuration?
    private static final String CLEAR_MAIN = "Clear";
    private static final String INTEGRATE_MAIN = "Integrate";
    private static final String ROW_SHIFT_R_MAIN = "RowShiftR";
    private static final String ROW_SHIFT_F_MAIN = "RowShiftF";
    private static final String READ_MAIN = "Read";
    private static final String PSEUDO_READ_MAIN = "PsuedoRead";
    private static final String[] MAINS = {CLEAR_MAIN, INTEGRATE_MAIN, ROW_SHIFT_R_MAIN, ROW_SHIFT_F_MAIN, READ_MAIN};

    private static final String CLEAR_COUNT_PARAMETER = "ClearCount";
    private static final String SHIFT_COUNT_PARAMETER = "ShiftCount";
    private static final String[] PARAMETERS = {CLEAR_COUNT_PARAMETER, SHIFT_COUNT_PARAMETER};

    private final List<SequencerProc> REBs = new ArrayList();
    private String imageName;
    private final FocalPlaneSubsystem subsys;
    private final FocalPlaneConfig focalPlaneConfig;
    private FPGA2Model model;
    
    Sequencers(FocalPlaneSubsystem subsys, FocalPlaneConfig focalPlaneConfig) {
        this.subsys = subsys;
        this.focalPlaneConfig = focalPlaneConfig;
    }

    void add(SequencerProc seq) {
        REBs.add(seq);
    }

    public void clear(int nClears) throws REBException, RaftException {
        setMain(CLEAR_MAIN);
        setParameter(CLEAR_COUNT_PARAMETER, nClears);
        run(false, FocalPlaneState.READING_OUT);
    }

    public void startIntegration(String imageName) throws REBException, RaftException {
        this.imageName = imageName;
        setMain(INTEGRATE_MAIN);
        run(false, FocalPlaneState.INTEGRATING);
    }

    public void rowShift(int nRows) throws REBException, RaftException {
        if (nRows == 0) {
            return;
        } else if (nRows > 0) {
            setMain(ROW_SHIFT_F_MAIN);
        } else {
            setMain(ROW_SHIFT_R_MAIN);
        }
        setParameter(SHIFT_COUNT_PARAMETER, Math.abs(nRows));
        run(false, FocalPlaneState.ROW_SHIFT);
    }

    public void stop() throws REBException {
        for (SequencerProc reb : REBs) {
            reb.sendStop();
        }
    }

    public void endIntegration(READOUT_MODE readout) throws REBException, RaftException {
        stop();
        waitForStop(Duration.ofSeconds(1));
        switch (readout) {
            case TRUE:
                setMain(READ_MAIN);
                run(true, FocalPlaneState.READING_OUT);
                break;
            case PSEUDO:
                setMain(PSEUDO_READ_MAIN);
                run(false, FocalPlaneState.READING_OUT);
                break;   
            case FALSE:
                // Nothing to do
        }
    }

    private void setMain(String mainName) throws REBException, RaftException {
        for (SequencerProc reb : REBs) {
            reb.setStart(mainName);
        }
    }

    private void setParameter(String parameter, int value) throws REBException, RaftException {
        for (SequencerProc reb : REBs) {
            reb.setParameter(parameter, value);
        }
    }

    private void run(boolean acquire, FocalPlaneState stateOnSuccess) throws REBException {
        if (acquire) {
            subsys.getGlobalProc().acquireImage(imageName);
        } else {
            for (SequencerProc reb : REBs) {
                reb.startSequencer();
            }
        }
        //TODO: What happens if above fails, state should not be left unchanged.
        setState(stateOnSuccess);
        doInBackground(() -> {
            try {
                // No way to tell how long we have to wait for infinite loop, 
                // so wait a somewhat arbitrary 1,000,000 seconds.
                waitForStop(Duration.ofMillis(1_000_000_000));
                setState(FocalPlaneState.QUIESCENT);
                LOG.info("Sequencers finished");
            } catch (REBException ex) {
                LOG.log(Level.SEVERE, "Error waiting for REBs", ex);
            }
        });
    }

    public void waitForStop(Duration timeout) throws REBException {
        for (SequencerProc reb : REBs) {
            // NOTE: waitDone throws a REBException it if times out
            reb.waitDone((int) timeout.toMillis());
            int errorAddr = reb.getErrorAddr();
            if (errorAddr != -1) {
                //TODO: Would be good to know which REB(s) had errors
                LOG.log(Level.SEVERE, "REB error register set {0}", errorAddr);
                //TODO: We should go into fault state here
            }
        }
    }

    /**
     * Fail fast if the sequencers loaded do not meet our minimal requirements
     *
     * @param seq The sequencer to test
     * @throws RaftException If any of the required mains or parameters are
     * missing
     */
    private void sanityCheck(FPGA2Model model) throws RaftException, REBException {
        Map<String, Integer> mainMap = model.getMainAddresses();
        for (String main : MAINS) {
            if (mainMap.get(main) == null) {
                throw new RaftException("Sequencer failed sanity check, missing main " + main);
            }
        }
        List<FPGA2Model.PointerInfo> pointers = model.getPointers();
        for (String parameter : PARAMETERS) {
            boolean found = false;
            for (FPGA2Model.PointerInfo pointer : pointers) {
                if (pointer.getName().equals(parameter)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new RaftException("Sequencer failed sanity check, missing parameter " + parameter);
            }
        }
        // TODO: The set of meta-data registers should be configurable
        int[] registers = computeMetaDataRegisters(SequencerData.imageRegNames, model);
        // TODO: Probably this is not the right place to do this
        if (subsys != null) {
           subsys.getGlobalProc().setRegisters(REB_TYPE_SCIENCE, registers);
        }
    }

    /**
     * Check that all required meta-data parameters are present in the sequencer.
     * @param metaData A map of required parameters (TODO: Should be just a list?)
     * @param model The sequencer mode to check
     * @return The list of addresses of the meta-data registers
     * @throws RaftException If one or more meta-data parameters are missing or invalid
     */
    static int[] computeMetaDataRegisters(String[] metaDataNames, FPGA2Model model) throws RaftException {
        int[] registers = new int[metaDataNames.length];
        Map<String, PointerInfo> pointerMap = model.getPointerMap();
        int k = 0;
        for (String metaName : metaDataNames) {
            PointerInfo pi = pointerMap.get(metaName);
            if (pi != null) {
                switch (pi.getKind()) {
                    case REPEAT_FUNCTION:
                        registers[k++] = pi.getAddress();
                        break;
                    case REPEAT_SUBROUTINE:
                        registers[k++] = pi.getAddress();
                        break;
                    default:
                        throw new RaftException("Parameter: " + metaName + " is of unsupported type " + pi.getKind());

                }
            } else {
                throw new RaftException("Required parameter not defined " + metaName);
            }
        }
        return registers;
    }

    void load() throws RaftException, REBException {
        try {
            InputStream input = BootstrapResourceUtils.getBootstrapResource(focalPlaneConfig.getSequencer());
            FPGA2ModelBuilder builder = new FPGA2ModelBuilder();
            FPGA2Model compiledModel = builder.compileFile(input);

            sanityCheck(compiledModel);
            
            this.model = compiledModel;

            for (SequencerProc reb : REBs) {
                try {
                   reb.loadSequencer(compiledModel);
                } catch (REBException ex) {
                   LOG.log(Level.WARNING, "Unable to load sequencer", ex);             
                }
            }
        } catch (IOException x) {
            RaftException re = new RaftException("Error reading sequencer ");
            re.initCause(x);
            throw re;
        }
    }

    private void doInBackground(Runnable run) {
        Thread t = new Thread(run);
        t.setName("Sequencer wait thread");
        t.start();
    }

    private void setState(FocalPlaneState focalPlaneState) {
        if (subsys != null) {
            subsys.updateAgentState(focalPlaneState);
        }
    }

    void validate(SequencerProc sequencer) {
        try {
            if (model != null) sequencer.loadSequencer(model);
        } catch (REBException ex) {
            //TODO: Enter fault state?
            LOG.log(Level.SEVERE, "Unable to load sequencer", ex);
        }
    }
}
