package org.lsst.ccs.subsystem.focalplane;

import java.io.File;
import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.daq.utilities.FitsHeaderKeywordData;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;

/**
 * Commands that are useful when running the focal plane subsystem with
 * scripting
 *
 * @author tonyj
 */
public class ScriptingCommands {

    private final FocalPlaneSubsystem subsys;

    ScriptingCommands(FocalPlaneSubsystem subsys) {
        this.subsys = subsys;
    }

    /**
     * Wait until all FITS files have been written.
     *
     * @param timeout The maximum duration to wait
     * @return The list of files which were written
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for FITS files to be saved", level = Command.NORMAL, autoAck = false)
    public FileList waitForFitsFiles(@Argument(defaultValue = "PT30S") Duration timeout) {
        return subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> subsys.getImageHandling().waitForFITSFiles(timeout));
    }

    /**
     * Wait until all images have been received from the DAQ (but not
     * necessarily written as FITS files)
     *
     * @param timeout The maximum duration to wait
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for image data to arrive", level = Command.NORMAL, autoAck = false)
    public void waitForImages(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                .action(() -> subsys.getImageHandling().waitForImages(timeout));
    }

    /**
     * Wait until the sequencer stops.
     * @param timeout 
     */
    @Command(type = Command.CommandType.QUERY, description = "Wait for sequencer to stop", level = Command.NORMAL, autoAck = false)
    public void waitForSequencer(@Argument(defaultValue = "PT30S") Duration timeout) {
        subsys.helper()
                .enterFaultOnException(true)
                .duration(timeout)
                // Note: dont use waitForStop, because that can return before the state 
                // has been updated, causing subsequent commands to fail
                .action(() -> subsys.waitFor((StateBundle t)
                -> t.isInState(FocalPlaneState.QUIESCENT) || t.isInState(FocalPlaneState.NEEDS_CLEAR),
                timeout.toMillis(), TimeUnit.MILLISECONDS));
    }

    /**
     * A simple extension of <code>List&lt;File&gt;</code> which add an extra utility method
     * for getting the common root directory of all the files in the list.
     */
    public static class FileList extends ConcurrentSkipListSet<File> {

        void addAll(List<String> files) {
            files.forEach((file) -> {
                add(new File(file));
            });
        }

        public File getCommonParentDirectory() {
            Iterator<File> iterator = this.iterator();
            if (!iterator.hasNext()) {
                return new File("/");
            }
            File result = iterator.next().getParentFile();
            while (iterator.hasNext()) {
                File file = iterator.next();
                while (!file.getAbsolutePath().startsWith(result.getAbsolutePath())) {
                    result = result.getParentFile();
                }
            }
            return result;
        }
    }
    
    //COMMANDS FOR SETTING HEADER KEYWORDS
    
    /**
     *  Set the value of Primary Header Keyword
     *
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.QUERY, description="Set a primary header keyword value")
    public void setPrimaryHeaderKeyword(String name, Serializable value) {
        setHeaderKeyword("primary", name, value);
    }

    /**
     *  Set the values of Primary Header Keywords
     *
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.QUERY, description="Set primary header keyword values")
    public void setPrimaryHeaderKeyword(Map<String, Serializable> headersMap) {
        setHeaderKeyword("primary", headersMap);
    }

    /**
     *  Set the value of a named HDU Header Keyword
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  name The name of the header keyword
     *  @param  value The value of the header keyword
     */
    @Command(type=Command.CommandType.QUERY, description="Set a header keyword value on a named HDU")
    public void setHeaderKeyword(String hduName, String name, Serializable value) {
        Map<String,Serializable> map = new HashMap<>();
        map.put(name, value);
        setHeaderKeyword(hduName, map);
    }

    /**
     *  Set the values of a named HDU Header Keywords
     *
     *  @param  hduName The name of the HDU on which to set the header keyword
     *  @param  headersMap The map of header keyword name, value pairs to be set
     */
    @Command(type=Command.CommandType.QUERY, description="Set header keyword values on a named HDU")
    public void setHeaderKeyword(String hduName, Map<String, Serializable> headersMap) {
        FitsHeaderKeywordData data = new FitsHeaderKeywordData();
        for ( Map.Entry<String,Serializable> e : headersMap.entrySet() ) {
            data.addHeaderKeywordValue(hduName, e.getKey(), e.getValue(), true);
        }
        subsys.publishSubsystemDataOnStatusBus(data.getKeyValueData());
    }

    
    
}
